/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.buffer;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentFactory;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.buffer.BufferBuilder;
import org.apache.flink.runtime.io.network.buffer.BufferBuilderTestUtils;
import org.apache.flink.runtime.io.network.buffer.BufferConsumer;
import org.apache.flink.runtime.io.network.buffer.BufferRecycler;
import org.apache.flink.runtime.io.network.buffer.FreeingBufferRecycler;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;

class BufferBuilderAndConsumerTest {
    private static final int BUFFER_INT_SIZE = 10;
    private static final int BUFFER_SIZE = 40;

    BufferBuilderAndConsumerTest() {
    }

    @Test
    void referenceCounting() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        BufferConsumer bufferConsumer = bufferBuilder.createBufferConsumer();
        Assertions.assertThat((int)bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(1, 2, 3))).isEqualTo(12);
        bufferBuilder.close();
        Buffer buffer = bufferConsumer.build();
        Assertions.assertThat((boolean)buffer.isRecycled()).isFalse();
        buffer.recycleBuffer();
        Assertions.assertThat((boolean)buffer.isRecycled()).isFalse();
        bufferConsumer.close();
        Assertions.assertThat((boolean)buffer.isRecycled()).isTrue();
    }

    @Test
    void append() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        BufferConsumer bufferConsumer = bufferBuilder.createBufferConsumer();
        int[] intsToWrite = new int[]{0, 1, 2, 3, 42};
        ByteBuffer bytesToWrite = BufferBuilderTestUtils.toByteBuffer(intsToWrite);
        Assertions.assertThat((int)bufferBuilder.appendAndCommit(bytesToWrite)).isEqualTo(bytesToWrite.limit());
        Assertions.assertThat((int)bytesToWrite.position()).isEqualTo(bytesToWrite.limit());
        Assertions.assertThat((boolean)bufferBuilder.isFull()).isFalse();
        BufferBuilderTestUtils.assertContent(bufferConsumer, intsToWrite);
    }

    @Test
    void multipleAppends() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        BufferConsumer bufferConsumer = bufferBuilder.createBufferConsumer();
        bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(0, 1));
        bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(2));
        bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(3, 42));
        BufferBuilderTestUtils.assertContent(bufferConsumer, 0, 1, 2, 3, 42);
    }

    @Test
    void multipleNotCommittedAppends() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        BufferConsumer bufferConsumer = bufferBuilder.createBufferConsumer();
        bufferBuilder.append(BufferBuilderTestUtils.toByteBuffer(0, 1));
        bufferBuilder.append(BufferBuilderTestUtils.toByteBuffer(2));
        bufferBuilder.append(BufferBuilderTestUtils.toByteBuffer(3, 42));
        BufferBuilderTestUtils.assertContent(bufferConsumer, new int[0]);
        bufferBuilder.commit();
        BufferBuilderTestUtils.assertContent(bufferConsumer, 0, 1, 2, 3, 42);
    }

    @Test
    void appendOverSize() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        BufferConsumer bufferConsumer = bufferBuilder.createBufferConsumer();
        ByteBuffer bytesToWrite = BufferBuilderTestUtils.toByteBuffer(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 42);
        Assertions.assertThat((int)bufferBuilder.appendAndCommit(bytesToWrite)).isEqualTo(40);
        Assertions.assertThat((boolean)bufferBuilder.isFull()).isTrue();
        BufferBuilderTestUtils.assertContent(bufferConsumer, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9);
        bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        bufferConsumer = bufferBuilder.createBufferConsumer();
        Assertions.assertThat((int)bufferBuilder.appendAndCommit(bytesToWrite)).isEqualTo(4);
        Assertions.assertThat((boolean)bufferBuilder.isFull()).isFalse();
        BufferBuilderTestUtils.assertContent(bufferConsumer, 42);
    }

    @Test
    void creatingBufferConsumerTwice() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        bufferBuilder.createBufferConsumer();
        Assertions.assertThatThrownBy(() -> ((BufferBuilder)bufferBuilder).createBufferConsumer()).isInstanceOf(IllegalStateException.class);
    }

    @Test
    void copy() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        BufferConsumer bufferConsumer1 = bufferBuilder.createBufferConsumer();
        bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(0, 1));
        BufferConsumer bufferConsumer2 = bufferConsumer1.copy();
        bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(2));
        BufferBuilderTestUtils.assertContent(bufferConsumer1, 0, 1, 2);
        BufferBuilderTestUtils.assertContent(bufferConsumer2, 0, 1, 2);
        BufferConsumer bufferConsumer3 = bufferConsumer1.copy();
        bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(3, 42));
        BufferConsumer bufferConsumer4 = bufferConsumer1.copy();
        BufferBuilderTestUtils.assertContent(bufferConsumer1, 3, 42);
        BufferBuilderTestUtils.assertContent(bufferConsumer2, 3, 42);
        BufferBuilderTestUtils.assertContent(bufferConsumer3, 3, 42);
        BufferBuilderTestUtils.assertContent(bufferConsumer4, 3, 42);
    }

    @Test
    void buildEmptyBuffer() {
        try (BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);){
            Buffer buffer = BufferBuilderTestUtils.buildSingleBuffer(bufferBuilder);
            Assertions.assertThat((int)buffer.getSize()).isZero();
            BufferBuilderTestUtils.assertContent(buffer, FreeingBufferRecycler.INSTANCE, new int[0]);
        }
    }

    @Test
    void buildingBufferMultipleTimes() {
        try (BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
             BufferConsumer bufferConsumer = bufferBuilder.createBufferConsumer();){
            bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(0, 1));
            bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(2));
            BufferBuilderTestUtils.assertContent(bufferConsumer, 0, 1, 2);
            bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(3, 42));
            bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(44));
            BufferBuilderTestUtils.assertContent(bufferConsumer, 3, 42, 44);
            ArrayList<Integer> originalValues = new ArrayList<Integer>();
            while (!bufferBuilder.isFull()) {
                bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(1337));
                originalValues.add(1337);
            }
            BufferBuilderTestUtils.assertContent(bufferConsumer, originalValues.stream().mapToInt(Integer::intValue).toArray());
        }
    }

    @Test
    void emptyIsFinished() {
        BufferBuilderAndConsumerTest.testIsFinished(0);
    }

    @Test
    void partiallyFullIsFinished() {
        BufferBuilderAndConsumerTest.testIsFinished(5);
    }

    @Test
    void fullIsFinished() {
        BufferBuilderAndConsumerTest.testIsFinished(10);
    }

    @Test
    void testWritableBytes() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        Assertions.assertThat((int)bufferBuilder.getWritableBytes()).isEqualTo(bufferBuilder.getMaxCapacity());
        ByteBuffer byteBuffer = BufferBuilderTestUtils.toByteBuffer(1, 2, 3);
        bufferBuilder.append(byteBuffer);
        Assertions.assertThat((int)bufferBuilder.getWritableBytes()).isEqualTo(bufferBuilder.getMaxCapacity() - byteBuffer.position());
        Assertions.assertThat((int)bufferBuilder.getWritableBytes()).isEqualTo(bufferBuilder.getMaxCapacity() - byteBuffer.position());
    }

    @Test
    void testWritableBytesWhenFull() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        bufferBuilder.append(BufferBuilderTestUtils.toByteBuffer(new int[bufferBuilder.getMaxCapacity()]));
        Assertions.assertThat((int)bufferBuilder.getWritableBytes()).isZero();
    }

    @Test
    void recycleWithoutConsumer() {
        CountedRecycler recycler = new CountedRecycler();
        BufferBuilder bufferBuilder = new BufferBuilder(MemorySegmentFactory.allocateUnpooledSegment((int)40), (BufferRecycler)recycler);
        bufferBuilder.close();
        Assertions.assertThat((int)recycler.recycleInvocationCounter).isOne();
    }

    @Test
    void recycleConsumerAndBufferBuilder() {
        CountedRecycler recycler = new CountedRecycler();
        BufferBuilder bufferBuilder = new BufferBuilder(MemorySegmentFactory.allocateUnpooledSegment((int)40), (BufferRecycler)recycler);
        BufferConsumer bufferConsumer = bufferBuilder.createBufferConsumer();
        bufferBuilder.close();
        Assertions.assertThat((int)recycler.recycleInvocationCounter).isZero();
        bufferConsumer.close();
        Assertions.assertThat((int)recycler.recycleInvocationCounter).isOne();
    }

    @Test
    void trimToAvailableSize() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        Assertions.assertThat((int)bufferBuilder.getMaxCapacity()).isEqualTo(40);
        bufferBuilder.trim(20);
        Assertions.assertThat((int)bufferBuilder.getMaxCapacity()).isEqualTo(20);
        bufferBuilder.trim(0);
        Assertions.assertThat((int)bufferBuilder.getMaxCapacity()).isZero();
    }

    @Test
    void trimToNegativeSize() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        Assertions.assertThat((int)bufferBuilder.getMaxCapacity()).isEqualTo(40);
        bufferBuilder.trim(-1);
        Assertions.assertThat((int)bufferBuilder.getMaxCapacity()).isZero();
    }

    @Test
    void trimToSizeLessThanWritten() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        Assertions.assertThat((int)bufferBuilder.getMaxCapacity()).isEqualTo(40);
        bufferBuilder.append(BufferBuilderTestUtils.toByteBuffer(1, 2, 3));
        bufferBuilder.trim(4);
        Assertions.assertThat((int)bufferBuilder.getMaxCapacity()).isEqualTo(12);
    }

    @Test
    void trimToSizeGreaterThanMax() {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        Assertions.assertThat((int)bufferBuilder.getMaxCapacity()).isEqualTo(40);
        bufferBuilder.trim(41);
        Assertions.assertThat((int)bufferBuilder.getMaxCapacity()).isEqualTo(40);
    }

    private static void testIsFinished(int writes) {
        BufferBuilder bufferBuilder = BufferBuilderTestUtils.createEmptyBufferBuilder(40);
        BufferConsumer bufferConsumer = bufferBuilder.createBufferConsumer();
        for (int i = 0; i < writes; ++i) {
            Assertions.assertThat((int)bufferBuilder.appendAndCommit(BufferBuilderTestUtils.toByteBuffer(42))).isEqualTo(4);
        }
        int expectedWrittenBytes = writes * 4;
        Assertions.assertThat((boolean)bufferBuilder.isFinished()).isFalse();
        Assertions.assertThat((boolean)bufferConsumer.isFinished()).isFalse();
        Assertions.assertThat((int)bufferConsumer.getWrittenBytes()).isZero();
        bufferConsumer.build();
        Assertions.assertThat((boolean)bufferBuilder.isFinished()).isFalse();
        Assertions.assertThat((boolean)bufferConsumer.isFinished()).isFalse();
        Assertions.assertThat((int)bufferConsumer.getWrittenBytes()).isEqualTo(expectedWrittenBytes);
        int actualWrittenBytes = bufferBuilder.finish();
        Assertions.assertThat((int)actualWrittenBytes).isEqualTo(expectedWrittenBytes);
        Assertions.assertThat((boolean)bufferBuilder.isFinished()).isTrue();
        Assertions.assertThat((boolean)bufferConsumer.isFinished()).isFalse();
        Assertions.assertThat((int)bufferConsumer.getWrittenBytes()).isEqualTo(expectedWrittenBytes);
        actualWrittenBytes = bufferBuilder.finish();
        Assertions.assertThat((int)actualWrittenBytes).isEqualTo(expectedWrittenBytes);
        Assertions.assertThat((boolean)bufferBuilder.isFinished()).isTrue();
        Assertions.assertThat((boolean)bufferConsumer.isFinished()).isFalse();
        Assertions.assertThat((int)bufferConsumer.getWrittenBytes()).isEqualTo(expectedWrittenBytes);
        Assertions.assertThat((int)bufferConsumer.build().getSize()).isZero();
        Assertions.assertThat((boolean)bufferBuilder.isFinished()).isTrue();
    }

    private static class CountedRecycler
    implements BufferRecycler {
        int recycleInvocationCounter;

        private CountedRecycler() {
        }

        public void recycle(MemorySegment memorySegment) {
            ++this.recycleInvocationCounter;
            memorySegment.free();
        }
    }
}

