/**
 * Copyright 2022 N5 Technologies, Inc
 *
 * This product includes software developed at N5 Technologies, Inc
 * (http://www.n5corp.com/) as well as software licenced to N5 Technologies,
 * Inc under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding
 * copyright ownership.
 *
 * N5 Technologies licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.neeve.perf.serialization.rumi.quark.serial;

import java.nio.ByteBuffer;

import java.io.UnsupportedEncodingException;

import com.neeve.io.IOBuffer;
import com.neeve.lang.XIntSequence;
import com.neeve.lang.XStringDeserializer;
import com.neeve.perf.serialization.Provider;
import com.neeve.perf.serialization.rumi.quark.*;
import com.neeve.sma.MessageView;
import com.neeve.util.UtlTime;

public class CarBenchmark implements Provider<Car> {
    final public static class CarDeserializationCallback implements Car.Deserializer.Callback {
        final private byte[] tempBuffer = new byte[128];
        final private int[] tempIntBuffer = new int[128];
        final private EngineDeserializationCallback engineDeserializationCallback = new EngineDeserializationCallback();
        final private FuelFigureDeserializationCallback fuelFigureDeserializationCallback = new FuelFigureDeserializationCallback();
        final private PerformanceFigureDeserializationCallback performanceFigureDeserializationCallback = new PerformanceFigureDeserializationCallback();
        long ts;

        @Override
        public void handleTimestamp(long val) {
            ts = val;
        }

        @Override
        public void handleBlock2(Car.Block2Deserializer val) {
            val.getSerialNumber();
            val.getModelYear();
            val.getAvailable();
            val.getCode();
            val.getVehicleCode().getTo(tempBuffer, 0);
        }
        @Override
        public void handleSomeNumbers(XIntSequence val) {
            final com.neeve.lang.XIntIterator iterator = val.iterator();
            while (iterator.hasNext()) {
                iterator.next();
            }
        }
        @Override
        public void handleExtras(Car.ExtrasDeserializer val) {
            final com.neeve.lang.XIterator<OptionalExtras> iterator = val.iterator();
            while (iterator.hasNext()) {
                iterator.next();
            }
        }
        @Override
        public void handleEngine(Engine.Deserializer val) { val.run(engineDeserializationCallback);}
        @Override
        public void handlePerformanceFigure(PerformanceFigure.Deserializer val) { val.run(performanceFigureDeserializationCallback);}
        @Override
        public void handleFuelFigure(FuelFigure.Deserializer val) { val.run(fuelFigureDeserializationCallback);}
        @Override
        public void handleManufacturer(XStringDeserializer val) { val.getTo(tempBuffer, 0);}
        @Override
        public void handleModel(XStringDeserializer val) { val.getTo(tempBuffer, 0);}
    }

    final private static class EngineDeserializationCallback implements Engine.Deserializer.Callback {
        final private byte[] tempBuffer = new byte[128];

        @Override
        final public void handleBlock1(Engine.Block1Deserializer val) {
            val.getCapacity();
            val.getNumCylinders();
            // Max RPM is supposed to be a constant and so fetching it here with Quark would do more work than other encoding types (since Quark does not support constants)
            val.getManufacturerCode().getTo(tempBuffer, 0);
            // Fuel is supposed to be a constant and so fetching it here with Quark would do more work than other encoding types (since Quark does not support constants)
        }
    }

    final private static class FuelFigureDeserializationCallback implements FuelFigure.Deserializer.Callback {
        final private byte[] tempBuffer = new byte[128];

        @Override
        final public void handleBlock1(FuelFigure.Block1Deserializer val) {
            val.getSpeed();
            val.getMpg();
        }
    }

    final private static class PerformanceFigureDeserializationCallback implements PerformanceFigure.Deserializer.Callback {
        final private AccelerationDeserializationCallback accelerationDeserializationCallback = new AccelerationDeserializationCallback();

        @Override
        final public void handleOctaneRating(byte val) { }
        @Override
        final public void handleAcceleration(Acceleration.Deserializer val) { val.run(accelerationDeserializationCallback);}
    }

    final private static class AccelerationDeserializationCallback implements Acceleration.Deserializer.Callback {
        public void handleBlock1(Acceleration.Block1Deserializer val) {
            val.getMph();
            val.getSeconds();
        }
    }

    private static final byte[] MANUFACTURER;
    private static final byte[] MODEL;
    private static final byte[] ENG_MAN_CODE;
    private static final byte[] VEHICLE_CODE;
    private static final int[] SOME_NUMBERS;
    private static final OptionalExtras[] EXTRAS;
    static {
        try {
            MANUFACTURER = "MANUFACTURER".getBytes("UTF-8");
            MODEL = "MODEL".getBytes("UTF-8");
            ENG_MAN_CODE = "abc".getBytes("UTF-8");
            VEHICLE_CODE = "abcdef".getBytes("UTF-8");
            SOME_NUMBERS = new int[] { 0, 1, 2, 3, 4 };
            EXTRAS = new OptionalExtras[] { OptionalExtras.sportsPack, OptionalExtras.sunRoof };
        }
        catch (final UnsupportedEncodingException ex) {
            throw new RuntimeException(ex);
        }
    }

    final private Car.Serializer carSerializer = Car.Serializer.create();
    final private Car.Deserializer carDeserializer = Car.Deserializer.create();
    final private CarDeserializationCallback cb = new CarDeserializationCallback();
    final private IOBuffer encodeBuffer = IOBuffer.create(1024);
    private int encodedLength;
    final private IOBuffer decodeBuffer = IOBuffer.create(1024);
    final private int decodeLength;
    {
        decodeLength = serializeTo(decodeBuffer);
    }

    final public static int serializeTo(final Car.Serializer carSerializer) {
        Engine.Serializer engineSerializer;
        PerformanceFigure.Serializer performanceFigureSerializer;
        FuelFigure.Serializer fuelFigureSerializer;
        Acceleration.Serializer accelerationSerializer;

        carSerializer.block2()
            .code(Code.A)
            .modelYear((short)2005)
            .serialNumber(12345)
            .available(BooleanType.T)
            .vehicleCode(VEHICLE_CODE);

        carSerializer.someNumbers(SOME_NUMBERS)
            .extras(EXTRAS);

        engineSerializer = carSerializer.engine();
        engineSerializer.block1()
            .capacity((short)4200)
            .numCylinders((byte)8)
            .manufacturerCode(ENG_MAN_CODE);
        engineSerializer.done();

        fuelFigureSerializer = carSerializer.fuelFigure();
        fuelFigureSerializer.block1()
            .speed((short)30)
            .mpg(35.9f);
        fuelFigureSerializer.done();
        fuelFigureSerializer = carSerializer.fuelFigure();
        fuelFigureSerializer.block1()
            .speed((short)55)
            .mpg(49.0f);
        fuelFigureSerializer.done();
        fuelFigureSerializer.block1()
            .speed((short)75)
            .mpg(40.0f);
        fuelFigureSerializer.done();

        performanceFigureSerializer = carSerializer.performanceFigure().octaneRating((byte)95);
        accelerationSerializer = performanceFigureSerializer.acceleration();
        accelerationSerializer.block1()
            .mph((short)30)
            .seconds(4.0f);
        accelerationSerializer.done();
        accelerationSerializer = performanceFigureSerializer.acceleration();
        accelerationSerializer.block1()
            .mph((short)60)
            .seconds(7.5f);
        accelerationSerializer.done();
        accelerationSerializer = performanceFigureSerializer.acceleration();
        accelerationSerializer.block1()
            .mph((short)100)
            .seconds(12.2f);
        accelerationSerializer.done();
        performanceFigureSerializer.done();

        performanceFigureSerializer = carSerializer.performanceFigure().octaneRating((byte)99);
        accelerationSerializer = performanceFigureSerializer.acceleration();
        accelerationSerializer.block1()
            .mph((short)30)
            .seconds(3.8f);
        accelerationSerializer.done();
        accelerationSerializer = performanceFigureSerializer.acceleration();
        accelerationSerializer.block1()
            .mph((short)60)
            .seconds(7.1f);
        accelerationSerializer.done();
        accelerationSerializer = performanceFigureSerializer.acceleration();
        accelerationSerializer.block1()
            .mph((short)100)
            .seconds(11.8f);
        accelerationSerializer.done();
        performanceFigureSerializer.done();

        return carSerializer.manufacturer(MANUFACTURER)
               .model(MODEL)
               .done();
    }

    final public static int serializeTo(final Car.Serializer carSerializer, final IOBuffer buffer) {
        return serializeTo(carSerializer.init(buffer));
    }

    final private int serializeTo(final IOBuffer buffer) {
        return serializeTo(carSerializer, buffer);
    }

    final private int serializeTo(final Car car) {
        return serializeTo(car.serializer(1024).timestamp(UtlTime.now()));
    }

    final private void deserializeFrom(final IOBuffer buffer, final int len) {
        carDeserializer.init(buffer, 0, len).run(cb);
    }

    final private void deserializeFrom(final Car car) {
        car.deserializer().run(cb);
    }

    @Override
    public String name() {
        return "rumi.quark.serial";
    }

    @Override
    public short vfid() {
        return com.neeve.perf.serialization.rumi.quark.MessageFactory.VFID;
    }

    @Override
    public short otype() {
        return com.neeve.perf.serialization.rumi.quark.MessageFactory.ID_Car;
    }

    @Override
    public int encoding() {
        return MessageView.ENCODING_TYPE_QUARK;
    }

    @Override
    public Car create(final boolean encode) {
        final Car car = Car.create();
        if (encode) {
            encode(car);
        }
        return car;
    }

    @Override
    public void prepareToEncode() {
    }

    @Override
    public void encode(final Car car) {
        encodedLength = serializeTo(car);
    }

    @Override
    public void encode() {
        encodedLength = serializeTo(encodeBuffer);
    }

    @Override
    public int encodedLength() {
        return encodedLength;
    }

    @Override
    public void postEncode() {
    }

    @Override
    public void prepareToDecode() {
    }

    @Override
    public void decode(final Car car) {
        deserializeFrom(car);
    }

    @Override
    public void decode() {
        deserializeFrom(decodeBuffer, decodeLength);
    }

    @Override
    public int decodedLength() {
        return decodeLength;
    }

    @Override
    public void postDecode() {
    }

    @Override
    public long dispose(final MessageView view) {
        final Car car = (Car)view;
        try {
            // note: the act of getting the timestamp will deserialize the entire payload
            //       if present i.e. of sender was configured to operate with encode=true
            cb.ts = 0l;
            decode(car);
            return cb.ts;
        }
        finally { 
            car.dispose();
        }
    }
}
