/**
 * Copyright 2022 N5 Technologies, Inc
 *
 * This product includes software developed at N5 Technologies, Inc
 * (http://www.n5corp.com/) as well as software licenced to N5 Technologies,
 * Inc under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding
 * copyright ownership.
 *
 * N5 Technologies licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.neeve.tools;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TimeZone;
import java.util.TreeMap;

import com.eaio.uuid.UUID;
import com.google.common.base.Stopwatch;
import com.google.common.base.Strings;
import com.neeve.ci.ManifestProductInfo;
import com.neeve.config.Config;
import com.neeve.lang.XLinkedHashMap;
import com.neeve.ods.IStoreBinding;
import com.neeve.ods.IStoreObjectFactory;
import com.neeve.ods.OdsException;
import com.neeve.ods.StoreCommitEntry;
import com.neeve.ods.StoreDescriptor;
import com.neeve.ods.StoreObjectFactoryRegistry;
import com.neeve.pkt.PktFactory;
import com.neeve.pkt.PktPacket;
import com.neeve.pkt.PktSubheaderODS;
import com.neeve.pkt.log.PktRecoveryLog;
import com.neeve.pkt.log.PktRecoveryLog.FileOpenMode;
import com.neeve.query.QueryEngine.BackgroundIndexingPolicy;
import com.neeve.query.QueryException;
import com.neeve.query.QueryRepository;
import com.neeve.query.impl.QueryIndexableRepository;
import com.neeve.query.impl.util.UtlQueryResultFormatter;
import com.neeve.query.index.IdxField;
import com.neeve.query.index.IdxIndex;
import com.neeve.rog.IRogChangeDataCaptureHandler;
import com.neeve.rog.IRogNode;
import com.neeve.rog.impl.RogGraphCollection;
import com.neeve.rog.impl.log.RogLogQueryEngineImpl;
import com.neeve.rog.impl.log.RogLogQueryFieldResolver;
import com.neeve.rog.log.RogLog;
import com.neeve.rog.log.RogLogReader.Entry;
import com.neeve.rog.log.RogLogCdcProcessor;
import com.neeve.rog.log.RogLogCompactor;
import com.neeve.rog.log.RogLogFactory;
import com.neeve.rog.log.RogLogMetadata;
import com.neeve.rog.log.RogLogQueryEngine;
import com.neeve.rog.log.RogLogReader;
import com.neeve.rog.log.RogLogQueryRepository;
import com.neeve.rog.log.RogLogQueryResultSet;
import com.neeve.rog.log.RogLogUtil;
import com.neeve.rog.log.RogLogUtil.JsonPrettyPrintStyle;
import com.neeve.tools.interactive.InteractiveTool;
import com.neeve.tools.interactive.InteractiveTool.BooleanPropertyParser;
import com.neeve.tools.interactive.InteractiveTool.ConfigProperty;
import com.neeve.tools.interactive.InteractiveTool.PropertyChangeHandler;
import com.neeve.tools.interactive.InteractiveTool.PropertyParser;
import com.neeve.tools.interactive.commands.AnnotatedCommand;
import com.neeve.trace.Tracer;
import com.neeve.trace.Tracer.Level;
import com.neeve.util.UtlBuffer;
import com.neeve.util.UtlFile;
import com.neeve.util.UtlReflection;
import com.neeve.util.UtlTableFormatter.Format;
import com.neeve.util.UtlTime;

import jargs.gnu.CmdLineParser;

/**
 * The "tlt" command handler
 */
final class TltCommand extends AbstractCommand {

    private static final String PROP_TLT_INIT_SCRIPTS = "nv.tlt.initScripts";

    enum Mode {
        QUERY, BROWSE;
    }

    enum MetadataPolicy {
        On, Off, Only;
    }

    enum DeserializationPolicy {
        Lazy, Eager
    }

    /*
     * Fullscan threshold parser
     */
    final public static class FullscanThresholdParser implements PropertyParser {

        /* (non-Javadoc)
         * @see com.neeve.tools.interactive.InteractiveTool.PropertyParser#parse(java.lang.String, java.lang.String)
        */
        @Override
        public Double parse(String threshold, String defaultValue) throws IllegalArgumentException {
            if (threshold == null) {
                threshold = defaultValue;
            }

            if (threshold == null) {
                return null;
            }

            Double value = Double.valueOf(threshold);

            if (value < 0 || value > 1) {
                throw new IllegalArgumentException("The threshold must be between 0.0 and 1.0");
            }

            return value;
        }
    }

    /*
     * 'open' command handler.
     */
    @AnnotatedCommand.Command(keywords = "open", description = "This command opens a transaction log or a set of transaction logs.")
    final public class Open extends AnnotatedCommand {
        @Option(shortForm = 'r', longForm = "repair", required = false, defaultValue = "false", description = "May be specified to indicate that the log should be repaired. Repair required that the file be opened in a writeable mode.")
        boolean repair;

        @Option(shortForm = 'm', longForm = "mode", required = false, defaultValue = "r", validOptions = { "r", "rw", "rws", "rwd" }, description = "Indicate the mode in which the log should be opened")
        String openMode;

        @Option(shortForm = 'p', longForm = "properties", required = false, defaultValue = "", description = "A comma separated list of key=value log open properties")
        String openProperties;

        @Argument(position = 1, name = "log", required = true, description = "The name of a log or directory containing log(s). When a directory is specified all logs in the directory are added. When the name of a log is specified, then the name can be a versioned or non-versioned named of the log with or without the ..log suffix")
        File log;

        @RemainingArgs(name = "additionalLogs", required = false, description = "Additional logs or directories to add")
        String[] additionalLogs;

        @Override
        final public void execute() throws Exception {
            // validate
            if (repair && openMode.equals("r")) {
                throw new IllegalArgumentException("The repair flag can't be used in read only mode. Use the -m flag to specify a writeable mode.");
            }

            // parse properties
            Properties properties = new Properties();
            if (openProperties != null && openProperties.length() > 0) {
                StringTokenizer tok = new StringTokenizer(openProperties, ",");
                while (tok.hasMoreTokens()) {
                    String keyValue = tok.nextToken();
                    String[] split = keyValue.split("=");
                    if (split.length != 2) {
                        throw new IllegalArgumentException("Invalid open property '" + keyValue + "' expected propName=propValue");
                    }
                    properties.put(split[0], split[1]);
                }
            }

            // open the main log
            openLogOrDirectory(log, repair, openMode, properties);

            // open additional logs
            for (String additional : additionalLogs) {
                openLogOrDirectory(new File(UtlFile.expandPath(additional)), repair, openMode, properties);
            }

            // switch to the newly opened log if not in query mode
            if (!inQueryMode()) {
                if (!log.isDirectory()) {
                    switchTo(mainLogFile(log));
                }
                else {
                    if (!openLogs.isEmpty()) {
                        switchTo(openLogs.keySet().iterator().next());
                    }
                }
            }
        }
    }

    /*
     * 'read' command handler.
     */
    @AnnotatedCommand.Command(keywords = "read", description = "Read a number of log entries.")
    final public class Read extends AnnotatedCommand {
        @Option(shortForm = 'i', longForm = "iterator", defaultValue = "false", description = "Whether to do a read using RogLogReader.next() to iterate over the entries. Otherwise, RogLogReader.read() would be used")
        boolean iterate;
        @Argument(name = "count", defaultValue = "1", required = false, position = 1, description = "The number of entries to read. A value of 0 will cause the log to be read to the end")
        int count;

        final private NumberFormat format = NumberFormat.getInstance();
        private volatile boolean interrupted = false;

        {
            format.setMaximumFractionDigits(2);
        }

        final private class ReadReceiver implements RogLogReader.ReadCallback {
            int numPuts;
            int numUpdates;
            int numRemoves;
            int numSends;
            int numMessages;

            public void onRead(final RogLogReader.Entry entry) {
                switch (entry.getEntryType()) {
                    case Put:
                        numPuts++;
                        break;

                    case Update:
                        numUpdates++;
                        break;

                    case Remove:
                        numRemoves++;
                        break;

                    case Send:
                        numSends++;
                        break;

                    case Message:
                        numMessages++;
                        break;
                }
            }
        }

        @Override
        final public void execute() throws Exception {
            interrupted = false;
            assertBrowseMode();
            assertOpen();
            if (count > 0) {
                console().info("Reading (" + (iterate ? "iter" : "async") + ") " + count + " entries...");
            }
            else {
                console().info("Reading (" + (iterate ? "iter" : "async") + ") " + count + " to end of log...");
            }
            final ReadReceiver receiver = new ReadReceiver();
            final int resolvedCount = count == 0 ? Integer.MAX_VALUE : count;
            if (iterate) {
                for (int i = 0 ; i < resolvedCount ; i++) {
                    final Entry entry = reader.next();
                    if (entry != null) {
                        try {
                            receiver.onRead(entry);
                        }
                        finally {
                            entry.dispose(); 
                        }
                    }
                    else {
                        break;
                    }
                }
            }
            else {
                reader.read(resolvedCount, receiver);
            }
            console().info("Read " + format.format(receiver.numMessages) + " Msgs, " + format.format(receiver.numSends) + " Sends, " + format.format(receiver.numPuts) + " Puts, " + format.format(receiver.numUpdates) + " Updates, " + format.format(receiver.numRemoves) + " Removes");
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public void interrupt(final Thread commandThread) {
            interrupted = true;
        }
    }

    /*
     * 'next' command handler.
     */
    @AnnotatedCommand.Command(keywords = "next", description = "Dump the contents of a number of the next entries in the log or query result to the console.")
    final public class Next extends AnnotatedCommand {
        @Option(shortForm = 'r', longForm = "raw", defaultValue = "false", description = "writes out the raw body")
        boolean raw;

        @Option(shortForm = 'd', longForm = "detailed", defaultValue = "false", description = "displays detailed dump of each entry (as opposed to a tabular format")
        boolean detailed;

        @Option(shortForm = 'c', longForm = "csv", defaultValue = "false", description = "indicates that csv mode should be used for brief format, otherwise a tabular format will be used")
        boolean csv;

        @Argument(position = 1, name = "count", defaultValue = "1", required = false, description = "Number of entries to display")
        int count;

        private volatile boolean interrupted = false;

        @Override
        final public void execute() throws Exception {
            interrupted = false;
            for (int i = 0; i < count && !interrupted; i++) {
                if (!inQueryMode()) {
                    assertOpen();
                    final RogLogReader.Entry entry = reader.next();
                    if (entry != null) {
                        try {
                            String rawHeader = "Unavailable";
                            String rawBody = null;
                            if (raw) {
                                rawHeader = entry.getPacket().getHeader().dump("");
                                rawBody = entry.getPacket().getBody().dump("");
                            }

                            try {
                                if (!detailed && i == 0) {
                                    console().info(RogLogReader.Entry.getHeaderRow(csv));
                                }
                                writeEntry(entry, detailed, csv, false, console().out());
                            }
                            finally {
                                if (raw) {
                                    console().info("[RAW HEADER]");
                                    console().info(rawHeader);
                                    console().info("[RAW BODY]");
                                    console().info(rawBody);
                                }
                            }
                        }
                        finally {
                            entry.dispose(); 
                        }
                    }
                    else {
                        console().info("EOF");
                        return;
                    }
                }
                else {
                    assertResultSet();
                    if (queryResults.next()) {
                        RogLogReader.Entry entry = queryResults.getRawResult();

                        String rawStr = null;
                        if (raw) {
                            if (entry != null) {
                                rawStr = entry.getPacket().getBody().dump("");
                            }
                        }

                        try {
                            if (!detailed) {
                                writeQueryResult(console().out(), i == 0, csv ? Format.CSV : Format.TABULAR);
                            }
                            else {
                                boolean showLog = (openLogs.size() > 1);
                                PrintStream out = console().out();
                                if (entry == null) {
                                    writeResultSetRow(queryResults, detailed, csv, showLog, out);
                                }
                                else {
                                    writeEntry(queryResults.getRawResult(), detailed, csv, showLog, out);
                                }
                            }
                        }
                        finally {
                            if (raw) {
                                console().info("");
                                console().info("[RAW BODY]");
                                if (rawStr == null) {
                                    console().info("Unavailable");
                                }
                                else {
                                    console().info(rawStr);
                                }
                            }
                        }
                    }
                    else {
                        console().info("No more results");
                        return;
                    }
                }
            }

            if (!detailed && inQueryMode()) {
                console().out().println(resultFooters.get(csv ? Format.CSV : Format.TABULAR));
            }
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public void interrupt(final Thread commandThread) {
            interrupted = true;
        }
    }

    /*
     * 'nextxn' command handler.
     */
    @AnnotatedCommand.Command(keywords = "nexttxn", description = "This command reads the next application transaction from the log and dumps its contents " +
            "to the console. An application transaction is a set of log entries grouped by the same." +
            "application transaction id. (browse mode only)")
    final public class NextTransaction extends AnnotatedCommand {
        @Option(shortForm = 'd', longForm = "detailed", defaultValue = "false", description = "displays detailed dump of each entry (as opposed to default brief tabular format")
        boolean detailed;

        @Option(shortForm = 'c', longForm = "csv", defaultValue = "false", description = "indicates that csv mode should be used for brief format, otherwise a tabular format will be used")
        boolean csv;

        @Option(shortForm = 'o', longForm = "sorted", defaultValue = "false", description = "DEPRECATED sorts transaction entries by placing replicated entries first followed by message entries")
        boolean sorted;

        private volatile boolean interrupted = false;

        @Override
        final public void execute() throws Exception {
            interrupted = false;
            assertBrowseMode();
            assertOpen();
            final RogLogReader.Transaction transaction = reader.nextTransaction();
            if (transaction != null) {
                final List<RogLogReader.Entry> entries = transaction.getEntries();
                console().info("Transaction #" + transaction.getId() + " {");
                if (!detailed) {
                    console().info(RogLogReader.Entry.getHeaderRow(csv));
                }
                for (RogLogReader.Entry entry : entries) {
                    if (interrupted) {
                        break;
                    }
                    writeEntry(entry, detailed, csv, false, console().out());
                }
                console().info("}");
            }
            else {
                console().info("EOF");
            }
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public void interrupt(final Thread commandThread) {
            interrupted = true;
        }
    }

    /*
     * 'skip' command handler.
     */
    @AnnotatedCommand.Command(keywords = "skip", description = "Skip over a number of log entries or query results.")
    final public class Skip extends AnnotatedCommand {
        @Argument(name = "count", defaultValue = "1", required = false, position = 1, description = "The number of entries to skip. A value of 0 will cause the log to be skipped to the end")
        int count;

        final private NumberFormat format = NumberFormat.getInstance();
        private volatile boolean interrupted = false;

        {
            format.setMaximumFractionDigits(2);
        }

        final private class SkipStatsCollector implements RogLogReader.SkipCallback {
            int numPuts;
            int numUpdates;
            int numRemoves;
            int numSends;
            int numMessages;

            public void onSkip(final RogLogReader.Entry.Type type) {
                switch (type) {
                    case Put:
                        numPuts++;
                        break;

                    case Update:
                        numUpdates++;
                        break;

                    case Remove:
                        numRemoves++;
                        break;

                    case Send:
                        numSends++;
                        break;

                    case Message:
                        numMessages++;
                        break;
                }
            }
        }

        @Override
        final public void execute() throws Exception {
            interrupted = false;
            if (inQueryMode()) {
                assertResultSet();
                int i = 0;
                for (i = 0; i < count & !interrupted; i++) {
                    if (!queryResults.next()) {
                        console().info("No more results.");
                        break;
                    }
                }
                console().info("Skipped " + i + " results");

            }
            else {
                assertOpen();
                if (count > 0) {
                    console().info("Skipping " + count + " entries...");
                }
                else {
                    console().info("Skipping to end of log...");
                }
                final SkipStatsCollector collector = new SkipStatsCollector();
                reader.skip(count == 0 ? Integer.MAX_VALUE : count, collector);
                console().info("Skipped " + format.format(collector.numMessages) + " Msgs, " + format.format(collector.numSends) + " Sends, " + format.format(collector.numPuts) + " Puts, " + format.format(collector.numUpdates) + " Updates, " + format.format(collector.numRemoves) + " Removes");
            }
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public void interrupt(final Thread commandThread) {
            interrupted = true;
        }
    }

    /*
     * 'skiptxn' command handler.
     */
    @AnnotatedCommand.Command(keywords = "skiptxn", description = "Skip over a set of log transactions. (browse mode only)")
    final public class SkipTransaction extends AnnotatedCommand {
        @Argument(name = "count", defaultValue = "1", required = false, position = 1, description = "The number of transactions to skip")
        int count;

        private volatile boolean interrupted = false;

        final private class SkipStatsCollector implements RogLogReader.TransactionSkipCallback {
            int numTransactions;

            public void onTransactionSkip(final RogLogReader.Transaction transaction) {
                numTransactions++;
            }
        }

        @Override
        final public void execute() throws Exception {
            interrupted = false;
            assertBrowseMode();
            assertOpen();
            final SkipStatsCollector collector = new SkipStatsCollector();
            for (int i = 0; i < count && !interrupted; i++) {
                reader.skipTransaction(1, collector);
            }
            console().info("Skipped " + collector.numTransactions + " Transactions");
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public void interrupt(final Thread commandThread) {
            interrupted = true;
        }
    }

    /*
     * 'rewind' command handler.
     */
    @AnnotatedCommand.Command(keywords = "rewind", description = "In browse mode rewinds the browser to the start of the log. In query mode rewinde the query results to the beginning")
    final public class Rewind extends AnnotatedCommand {
        @Override
        final public void execute() throws Exception {
            if (!inQueryMode()) {
                assertOpen();
                reader.rewind();
            }
            else {
                assertResultSet();
                queryResults.beforeFirst();
            }
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public boolean interruptable() {
            return false;
        }
    }

    /*
     * 'tail' command handler.
     */
    @AnnotatedCommand.Command(keywords = "tail", description = "Tails entries in a current log (browse mode only)")
    final public class Tail extends AnnotatedCommand {
        @Option(shortForm = 't', longForm = "transactions", defaultValue = "false", required = false, description = "Indicates that the log should tail completed transactions rather than entries")
        boolean transactions;

        @Option(shortForm = 's', longForm = "stats", defaultValue = "false", required = false, description = "Indicates that aggregate log stats should be displayed rather than entries")
        boolean stats;

        @Option(shortForm = 'o', longForm = "sorted", defaultValue = "false", description = "DEPRECATED: sorts transaction entries by placing replicated entries first followed by message entries")
        boolean sorted;

        @Option(shortForm = 'd', longForm = "detailed", defaultValue = "false", description = "displays detailed dump of each entry (as opposed to default brief tabular format")
        boolean detailed;

        @Option(shortForm = 'f', longForm = "follow", defaultValue = "false", description = "Continually follows the log file")
        boolean follow;

        @Option(shortForm = 'i', longForm = "interval", defaultValue = "1.0", description = "with --follow the interval to check for new entries in seconds")
        float intervalSeconds;

        @Option(shortForm = 'p', longForm = "noorphan", defaultValue = "false", description = "Skips orphans")
        boolean noOrphan;

        @Option(shortForm = 'c', longForm = "csv", defaultValue = "false", description = "indicates that csv mode should be used for brief format, otherwise a tabular format will be used")
        boolean csv;

        @Option(shortForm = 'n', longForm = "count", defaultValue = "10", description = "output the last N lines, instead of the last 10")
        int numEntries;

        private volatile boolean interrupted = false;

        @Override
        final public void execute() throws Exception {
            interrupted = false;
            assertBrowseMode();
            assertOpen();
            reader.rewind();

            long interval = (long)intervalSeconds * 1000;

            if (!stats && !detailed) {
                console().info(RogLogReader.Entry.getHeaderRow(csv));
            }

            if (stats) {
                console().info(RogLogReader.Stats.getHeaderRow());
            }

            boolean first = true;
            while (!interrupted) {
                if (transactions) {
                    if (first) {
                        RogLogReader.Stats stats = reader.computeStats();
                        first = false;

                        int numTxns = stats.getNumTransactions();
                        reader.rewind();

                        if (numTxns > numEntries) {
                            reader.skipTransaction(numTxns - numEntries, null);
                        }
                    }

                    RogLogReader.Transaction transaction = reader.nextTransaction();
                    if (transaction != null) {
                        if (transaction.getId() != -1) {
                            console().info("Transaction id=" + transaction.getId());
                        }
                        else if (noOrphan) {
                            continue;
                        }
                        else {
                            console().info("Orphaned Entry:");
                        }
                        if (!stats) {
                            final List<RogLogReader.Entry> entries = transaction.getEntries();
                            for (RogLogReader.Entry entry : entries) {
                                writeEntry(entry, detailed, csv, false, console().out());
                            }
                        }
                        else {
                            console().info(reader.getStats().toString());
                        }
                    }
                    else {
                        if (follow) {
                            Thread.sleep(Math.max(100, interval));
                        }
                        else {
                            return;
                        }
                    }
                }
                else {
                    if (first) {
                        RogLogReader.Stats stats = reader.computeStats();
                        first = false;

                        int count = stats.getNumEntries();
                        reader.rewind();

                        if (count > numEntries) {
                            reader.skipTransaction(count - numEntries, null);
                        }
                    }

                    final RogLogReader.Entry entry = reader.next();
                    if (entry != null) {
                        if (!stats) {
                            writeEntry(entry, detailed, csv, false, console().out());
                        }
                        else {
                            console().info(reader.getStats().toString());
                        }
                    }
                    else {
                        if (follow) {
                            Thread.sleep(Math.max(100, interval));
                        }
                        else {
                            return;
                        }
                    }
                }
            }
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public void interrupt(final Thread commandThread) {
            interrupted = true;
        }
    }

    /*
     * 'stats' command handler.
     */
    @AnnotatedCommand.Command(keywords = "stats", description = "Displays log file stats for the entries read so far. (browse mode only)")
    final public class Stats extends AnnotatedCommand {
        @Option(shortForm = 'a', longForm = "all", defaultValue = "false", description = "Reads to the end of the log and display all stats")
        boolean all;

        @Override
        final public void execute() throws Exception {
            assertBrowseMode();
            assertOpen();

            console().info(RogLogReader.Stats.getHeaderRow());
            if (all) {
                console().info(reader.computeStats().toString());
            }
            else {
                console().info(reader.getStats().toString());
            }
        }

        /**
         * Check if interruptibe
         */
        @Override
        public boolean interruptable() {
            return false;
        }

    }

    /*
     * 'switch' command handler.
     */
    @AnnotatedCommand.Command(keywords = "switch", description = "Switches between browse and query modes. Query mode allows queries against the open logs, whilst browse mode allows you to browse through the current log.")
    final public class Switch extends AnnotatedCommand {
        @Option(shortForm = 'l', longForm = "list", defaultValue = "false", description = "Indicates the list of options should be displayed")
        boolean list = false;

        @Argument(position = 1, name = "target", required = false, description = "The name of the log to switch to in browse mode, 'query' to switch to query mode or 'browse' to switch to browse mode.")
        String target;

        @Override
        final public void execute() throws Exception {
            if (list) {
                listOpen();
                console().info("You can use 'switch <logname>' to browse or 'switch query' to issue queries and/or view results");
            }

            if (target != null) {
                switchTo(target);
            }
        }

        public final boolean interruptable() {
            return false;
        }

    }

    /*
     * 'factory' command handler.
     */
    @AnnotatedCommand.Command(keywords = "factory", description = "Register a factory with the X runtime")
    final public class Factory extends AnnotatedCommand {
        @Argument(name = "factoryClass", position = 1, description = "The fully qualified factory class name")
        String factoryClass;

        final private String normalizeForV3(final String className) {
            return className.equals("com.neeve.server.mon.SrvMonFactory") ? "com.neeve.server.mon.SrvMonHeartbeatFactory" : (className.equals("com.neeve.rog.impl.RogRawMessageFactory") ? "com.neeve.rog.impl.RogPacketMessageFactory" : className);
        }

        /**
         * @param className adds the given factory 
         *  
         * @throws Exception If there is an error loading the factory
         */
        final public void doFactory(String className) throws Exception {
            try {
                final Class<?> clazz = Class.forName(normalizeForV3(className));
                Method createMethod = null;
                try {
                    Class<?>[] parameterTypes = new Class<?>[1];
                    parameterTypes[0] = Class.forName("java.util.Properties");
                    createMethod = clazz.getMethod("create", parameterTypes);
                }
                catch (ClassNotFoundException e) {
                    throw new InternalError("Failed to load java.util.Properties during instantiation of factory class");
                }
                catch (SecurityException e) {
                    throw new Exception("Invalid factory class [Access to instantiation method is denied]");
                }
                catch (NoSuchMethodException e) {
                    throw new Exception("Invalid factory class [Instantiation method could not be found]");
                }

                /*
                 * Instantiate
                 */
                IStoreObjectFactory factory = null;
                try {
                    try {
                        Object[] parameters = new Object[1];
                        parameters[0] = null;
                        factory = (IStoreObjectFactory)createMethod.invoke(null, parameters);
                        if (factory == null) {
                            throw new Exception("Invalid factory class [Instantiation method returned a null object]");
                        }
                        RogLogReader.registerFactory(factory);
                    }
                    catch (ClassCastException e) {
                        throw new Exception("Invalid factory class [Instantiation method did not return a valid factory");
                    }
                }
                catch (IllegalAccessException e) {
                    throw new Exception("Invalid factory class [Access to instantiation method is denied]");
                }
            }
            catch (Exception e) {
                System.err.println("Failed to load factory '" + className + "' [" + e.toString() + "]");
            }
        }

        @Override
        final public void execute() throws Exception {
            doFactory(factoryClass);
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public boolean interruptable() {
            return false;
        }
    }

    /*
     * 'factories' command handler.
     */
    @AnnotatedCommand.Command(keywords = "factories", description = "Register a set of factories with the X runtime")
    final public class Factories extends AnnotatedCommand {
        @Argument(name = "factoryFile", description = "The name of file containing factories", required = true, position = 1)
        File file;

        @Override
        final public void execute() throws Exception {
            doFactories(file);
        }

        public void doFactories(String factoriesFilename) throws Exception {
            doFactories(new File(UtlFile.expandPath(factoriesFilename)));
        }

        public void doFactories(File file) throws Exception {
            final Factory factory = new Factory();
            if (file.exists()) {
                final BufferedReader br = new BufferedReader(new FileReader(file));
                try {
                    String line;
                    while ((line = br.readLine()) != null) {
                        factory.doFactory(line.trim());
                    }
                }
                finally {
                    br.close();
                }
            }
            else {
                throw new Exception("File not found: " + file);
            }
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public boolean interruptable() {
            return false;
        }
    }

    @AnnotatedCommand.Command(keywords = "consume", description = "Consume the current query. Primarily used for performance testing.", hidden = true)
    final public class Consume extends AnnotatedCommand {
        private volatile boolean interrupted = false;

        @Override
        public void execute() throws Exception {
            interrupted = false;
            if (inQueryMode()) {
                assertResultSet();
                while (queryResults.next() && !interrupted) {
                    queryResults.getRawResult(); // just retrieve the Entry, do nothing with it
                }
            }
        }

        @Override
        public void interrupt(final Thread commandThread) {
            interrupted = true;
        }
    }

    @AnnotatedCommand.Command(keywords = "diag", description = "Sets diagnostic parameters.", hidden = true)
    final public class Diag extends AnnotatedCommand {
        //        @Option(shortForm = 'l', longForm = "list", defaultValue = "false", description = "Used to list current dianostic settings")
        //        boolean list;

        @Argument(position = 1, name = "parameter", required = false, validOptions = { "typeInference", "staticTypes" }, description = "The parameter to set")
        String parameter;

        @Argument(position = 2, name = "parameter", required = false, description = "The parameter value")
        String value;

        @Override
        public void execute() throws Exception {
            if (parameter != null) {

                if (parameter.equals("typeInference")) {
                    switchTo("query");
                    queryEngine.enableTypeInference(new BooleanPropertyParser().parse(value, "false"));
                }
                else {
                    switchTo("query");
                    queryEngine.enableStaticFields(new BooleanPropertyParser().parse(value, "false"));
                }

                console().info("Set " + parameter + " to " + value);
            }
        }
    }

    @AnnotatedCommand.Command(keywords = "pktdiag", description = "Test packet deserialization.", hidden = true)
    final public class PktDiag extends AnnotatedCommand {

        @Option(shortForm = 'v', longForm = "verbose", defaultValue = "false", description = "Whether to trace deserialization details.")
        boolean verbose = false;

        @Option(shortForm = 'o', longForm = "offset", defaultValue = "0", description = "The offset into the dump.")
        int offset = 0;

        @Option(shortForm = 'a', longForm = "all", defaultValue = "false", description = "Read all packets that can be read from the given offset.")
        boolean all = false;

        @Option(shortForm = 'e', longForm = "dumpEntry", defaultValue = "false", description = "Dumps the packet as a log entry (requires 'factories' to be registered ... see 'factories -h').")
        boolean dumpEntry = false;

        @RemainingArgs(name = "pktDumpHex", required = true, description = "The packet in hex form as dumped by UtlBuffer.dump.")
        String pktDump;

        @Override
        public void execute() throws Exception {
            ByteBuffer buffer = UtlBuffer.fromDump(pktDump);
            console().info("Decoded " + buffer.remaining() + " hex bytes. Attempting to deserialize packet at offset: " + offset);

            Tracer tracer = Tracer.create(verbose ? Level.DEBUG : Level.INFO);
            tracer.bind("pkt.diag");
            PktPacket packet = PktFactory.getInstance().createPacket(buffer, offset, buffer.limit() - offset);
            offset += packet.getSerializedLength();
            console().info("Successfully deserialized packet: " + packet);
            if (dumpEntry) {
                dumpStoreObject(packet);
            }
            try {
                packet.dispose();
            }
            catch (Throwable thrown) {}

            if (all) {
                while (true) {
                    packet = PktFactory.getInstance().createPacket(buffer, offset, buffer.limit() - offset);
                    if (packet == null) {
                        break;
                    }
                    offset += packet.getSerializedLength();
                    console().info("Successfully deserialized packet: " + packet);
                    if (dumpEntry) {
                        dumpStoreObject(packet);
                    }
                    try {
                        packet.dispose();
                    }
                    catch (Throwable t) {
                        error("Error disposing packet: " + t.getMessage(), t);
                        console().info("Continuing...");
                    }
                }
            }
        }

        private void dumpStoreObject(PktPacket packet) throws Exception {
            PktSubheaderODS odsHeader = packet.getHeader().getODSSubheader();
            if (odsHeader == null) {
                console().info("...not an ods object.");
                return;
            }

            RogLogReader.Entry entry = RogLogReader.Entry.create(null, packet, 0l, 0);
            StringWriter sw = new StringWriter();
            sw.flush();

            final IStoreObjectFactory factory = StoreObjectFactoryRegistry.getInstance().getObjectFactory(odsHeader.getObjectFactoryId());
            if (factory == null) {
                console().info("Factory for object with ofid=" + odsHeader.getObjectFactoryId() + " not found (did you register factories with 'factories' command?");
            }

            RogLogUtil.dumpLogEntryJson(entry, true, factory != null, all, jsonPrettyPrintStyle, sw);
            console().info("Entry:\n" + sw.toString());
            entry.dispose();
        }
    }

    /*
     * 'dump' command handler.
     */
    @AnnotatedCommand.Command(keywords = "dump", description = "Dump contents of the current log or query results")
    final public class Dump extends AnnotatedCommand {
        @Option(shortForm = 'd', longForm = "detailed", defaultValue = "false", description = "displays detailed dump of each entry (as opposed to default brief tabular format")
        boolean detailed;

        @Option(shortForm = 'r', longForm = "raw", defaultValue = "false", description = "writes out the entry in raw format")
        boolean raw;

        @Option(shortForm = 'g', longForm = "group", defaultValue = "false", description = "groups dumped entries by transaction")
        boolean group;

        @Option(shortForm = 'o', longForm = "sorted", defaultValue = "false", description = "DEPRECATED -sorts transaction entries by placing replicated entries first followed by message entries")
        boolean sorted;

        @Option(shortForm = 's', longForm = "state", defaultValue = "false", description = "build and dumps the application state graph")
        boolean state;

        @Option(shortForm = 'x', longForm = "skipEntries", defaultValue = "false", description = "skips displaying entries")
        boolean skipEntries;

        @Option(shortForm = 'n', longForm = "noOrphan", defaultValue = "false", description = "skips displaying entries without a transaction id")
        boolean noOrphan;

        @Option(shortForm = 'f', longForm = "force", defaultValue = "false", description = "when dumping to a file that already exists, forces overwrite of the file.")
        boolean force;

        @Option(shortForm = 'p', longForm = "packets", defaultValue = "false", description = "Indicates whether or not packets should be dumped instead of entries (browse mode only).")
        boolean packets;

        @Argument(name = "dumpFile", position = 1, required = false, description = "The output file to which to dump. Suffixing with .csv results in csv output and with .html result in html")
        File dumpFile;

        IStoreBinding store = null;
        RogGraphCollection collection = null;

        private volatile boolean interrupted = false;

        @Override
        final public void execute() throws Exception {
            interrupted = false;
            if (inQueryMode()) {
                if (skipEntries) {
                    throw new Exception("skipEntries flag doesn't apply to query results");
                }

                if (noOrphan) {
                    throw new Exception("noOrphan flag doesn't apply to query results");
                }

                if (state) {
                    throw new Exception("state flag doesn't apply to query results");
                }

                if (sorted) {
                    throw new Exception("sorted flag doesn't apply to query results");
                }

                if (group) {
                    throw new Exception("group flag doesn't apply to query results");
                }

                if (packets) {
                    throw new Exception("packet dump is not a valid option in query mode");
                }

                assertResultSet();

                // rewind
                queryResults.beforeFirst();

                Format format = Format.TABULAR;
                final BufferedWriter bw;
                if (dumpFile != null) {
                    if (dumpFile.exists()) {
                        if (force) {
                            dumpFile.delete();
                        }
                        else {
                            throw new Exception("File '" + dumpFile.getName() + "' already exists. Use '-f' to force overwrite");
                        }
                    }
                    if (dumpFile.getName().endsWith(".html")) {
                        format = Format.HTML;
                    }
                    else if (dumpFile.getName().endsWith(".csv")) {
                        format = Format.CSV;
                    }

                    bw = new BufferedWriter(new FileWriter(dumpFile));
                }
                else {
                    bw = new BufferedWriter(new PrintWriter(console().out()));
                }

                try {
                    boolean first = true;
                    while (queryResults.next() && !interrupted) {
                        String rawStr = null;
                        if (raw) {
                            RogLogReader.Entry entry = queryResults.getRawResult();
                            if (entry != null) {
                                rawStr = entry.getPacket().getBody().dump("");
                            }
                        }

                        try {
                            if (!detailed) {
                                writeQueryResult(bw, first, format);
                            }
                            else {
                                Entry entry = queryResults.getRawResult();
                                boolean csv = (format == Format.CSV);
                                boolean showLog = (openLogs.size() > 1);
                                if (entry == null) {
                                    writeResultSetRow(queryResults, detailed, csv, showLog, bw);
                                }
                                else {
                                    writeEntry(queryResults.getRawResult(), detailed, csv, showLog, bw);
                                }
                            }
                            first = false;
                        }
                        finally {
                            if (raw) {
                                switch (format) {
                                    case CSV:
                                        bw.append("\"[RAW BODY]: ");
                                        if (rawStr == null) {
                                            bw.append("Unavailable");
                                        }
                                        else {
                                            bw.append(rawStr);
                                        }
                                        bw.append("\"");
                                        bw.newLine();
                                        break;

                                    case HTML:
                                        bw.append("<tr><td colspan=\"100\"><pre style=\"white-space: pre-wrap;\">");
                                        bw.append("[RAW BODY]");
                                        bw.newLine();
                                        if (rawStr == null) {
                                            bw.append("Unavailable");
                                        }
                                        else {
                                            bw.append(rawStr);
                                        }
                                        bw.append("</pre></td></tr>");
                                        break;

                                    case TABULAR:
                                        bw.append("  [RAW BODY]");
                                        bw.newLine();
                                        if (rawStr == null) {
                                            bw.append("Unavailable");
                                        }
                                        else {
                                            bw.append(rawStr);
                                        }
                                        bw.newLine();
                                        break;

                                    default:
                                        break;
                                }
                            }
                        }
                    }

                    if (!detailed) {
                        String resultFooter = resultFooters.get(format);
                        if (resultFooter != null) {
                            bw.write(resultFooters.get(format));
                        }
                    }
                }
                finally {
                    bw.flush();
                    if (dumpFile != null) {
                        bw.close();
                    }
                }
            }
            else {
                assertOpen();
                final BufferedWriter bw;

                Format format = Format.TABULAR;
                if (dumpFile != null) {
                    if (dumpFile.exists()) {
                        if (force) {
                            dumpFile.delete();
                        }
                        else {
                            throw new Exception("File '" + dumpFile.getName() + "' already exists. Use '-f' to force overwrite.");
                        }
                    }

                    if (dumpFile.getName().endsWith(".html")) {
                        throw new Exception("HTML dump is only supported for query results");
                    }
                    else if (dumpFile.getName().endsWith(".csv")) {
                        format = Format.CSV;
                    }
                    bw = new BufferedWriter(new FileWriter(dumpFile));
                }
                else {
                    bw = new BufferedWriter(new PrintWriter(console().out()));
                }

                try {

                    if (packets) {
                        File logFile = reader.log().getLogFile();
                        PktRecoveryLog pktlog = PktRecoveryLog.create(logFile.getParent(), logFile.getName())
                                                              .setOpenMode(FileOpenMode.rw)
                                                              .setInitialLength(0)
                                                              .setPageSize(8192)
                                                              .setReadBufferSize(8192)
                                                              .setWriteBufferSize(8192);
                        pktlog.open(0, 0);

                        try {
                            pktlog.dump(bw);
                        }
                        finally {
                            pktlog.close();
                        }
                        return;
                    }

                    if (!skipEntries) {
                        reader.rewind();
                        bw.write("[LOG ENTRIES]");
                        bw.newLine();
                        if (group) {
                            RogLogReader.Transaction transaction;
                            if (!detailed) {
                                bw.append(RogLogReader.Entry.getHeaderRow(format == Format.CSV));
                                bw.newLine();
                                bw.newLine();
                            }
                            while ((transaction = reader.nextTransaction()) != null && !interrupted) {
                                if (noOrphan && transaction.getId() == -1) {
                                    continue;
                                }
                                for (RogLogReader.Entry entry : transaction.getEntries()) {
                                    String rawStr = null;
                                    if (raw) {
                                        rawStr = entry.getPacket().getBody().dump("");
                                    }

                                    try {
                                        writeEntry(entry, detailed, format == Format.CSV, false, bw);
                                    }
                                    finally {
                                        if (raw) {
                                            switch (format) {
                                                case CSV:
                                                    bw.append("\"[RAW BODY]: ");
                                                    if (rawStr == null) {
                                                        bw.append("Unavailable");
                                                    }
                                                    else {
                                                        bw.append(rawStr);
                                                    }
                                                    bw.append("\"");
                                                    bw.newLine();
                                                    break;
                                                case HTML:
                                                    bw.append("<tr><td colspan=\"100\"><pre style=\"white-space: pre-wrap;\">");
                                                    bw.append("[RAW BODY]");
                                                    bw.newLine();
                                                    if (rawStr == null) {
                                                        bw.append("Unavailable");
                                                    }
                                                    else {
                                                        bw.append(rawStr);
                                                    }
                                                    bw.append("</pre></td></tr>");
                                                    break;
                                                case TABULAR:
                                                    bw.append("  [RAW BODY]");
                                                    bw.newLine();
                                                    if (rawStr == null) {
                                                        bw.append("Unavailable");
                                                    }
                                                    else {
                                                        bw.append(rawStr);
                                                    }
                                                    bw.newLine();
                                                    break;
                                                default:
                                                    break;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        else {
                            RogLogReader.Entry entry;
                            if (!detailed) {
                                bw.append(RogLogReader.Entry.getHeaderRow(format == Format.CSV));
                                bw.newLine();
                            }
                            while ((entry = reader.next()) != null && !interrupted) {
                                if (noOrphan && entry.getTransactionId() == -1) {
                                    continue;
                                }

                                String rawStr = null;
                                if (raw) {
                                    rawStr = entry.getPacket().getBody().dump("");
                                }

                                try {
                                    writeEntry(entry, detailed, format == Format.CSV, false, bw);
                                }
                                finally {
                                    if (raw) {
                                        bw.append("[RAW BODY]");
                                        bw.newLine();
                                        if (rawStr == null) {
                                            bw.append("Unavailable");
                                        }
                                        else {
                                            bw.append(rawStr);
                                        }
                                        bw.newLine();
                                    }
                                }
                            }
                        }
                    }

                    if (state && !interrupted) {
                        String name = "LogReaderTool" + System.currentTimeMillis();
                        StoreDescriptor descriptor = StoreDescriptor.create("store");
                        collection = (RogGraphCollection)RogGraphCollection.create(name, descriptor, null, 0, 0);
                        store = collection.getStore();
                        reader.rewind();
                        //                        store.setReader(reader);
                        store.open();
                        try {
                            bw.newLine();
                            bw.newLine();
                            bw.write("[STATE DUMP]");
                            collection.dump(bw);
                            //                            store.dump(bw, 
                            //                                       metadataDisplay == MetadataPolicy.Only || metadataDisplay == MetadataPolicy.On, 
                            //                                       metadataDisplay != MetadataPolicy.Only,
                            //                                       filterUnsetFields, 
                            //                                       jsonPrettyPrintStyle);
                        }
                        finally {
                            store.close(0);
                        }
                    }
                }
                finally {
                    bw.flush();
                    if (dumpFile != null) {
                        bw.close();
                    }
                }
            }
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public void interrupt(final Thread commandThread) {
            interrupted = true;
        }
    }

    /*
     * 'rewrite' command handler.
     */
    @AnnotatedCommand.Command(keywords = { "rewrite", "writelog" }, hidden = false, description = "Rewrites the results of the current query as a new transaction log." +
            " Note that recovering from a rewritten transaction log is an inherently unsafe operation ... removing an event from the recovery stream" +
            " may result in an unrecoverable transaction log or recovery with inconsistent state. Extreme caution should be exercised when using a rewritten " +
            " transaction log in a production environment.")
    final public class Rewrite extends AnnotatedCommand {

        @Argument(name = "outputFolder", required = true, position = 1, description = "The output folder to which to write rewritten logs")
        File outputFolder;

        private volatile boolean interrupted = false;

        @Override
        final public void execute() throws Exception {
            assertQueryMode();
            assertResultSet();

            // Results may be accross multiple logs
            Map<String, RogLog> rewrittenLogs = new HashMap<String, RogLog>();

            // Rewritten results may change transaction boundaries so we don't write out packets
            // until we can confirm that the next packet begins a new transaction or not.
            XLinkedHashMap<String, PktPacket> pendingPackets = new XLinkedHashMap<String, PktPacket>();
            queryResults.beforeFirst();

            try {
                int recordsWritten = 0;
                while (queryResults.next() && !interrupted) {
                    RogLogReader.Entry entry = queryResults.getRawResult();

                    if (entry != null) {
                        String logName = entry.getLogName();
                        RogLog targetLog = rewrittenLogs.get(logName);
                        if (targetLog == null) {
                            console().info("Opening rewrite log '" + outputFolder + "/" + logName + ".log");
                            final Properties props = new Properties();
                            props.setProperty("detachedPersist", "true");
                            props.setProperty("storeRoot", outputFolder.getCanonicalPath());
                            props.setProperty("autoRepair", "true");
                            props.setProperty("logMode", "rw");
                            targetLog = RogLog.create(logName, props);
                            if (targetLog.getLogFile().exists()) {
                                console().info("Target rewrite log already exists (" + targetLog.getLogFile().getCanonicalPath() + ")... backing it up.");
                                String backup = targetLog.backupLog(true);
                                console().info("Target log backed up to '" + backup + "'");
                            }
                            targetLog.open();
                            rewrittenLogs.put(logName, targetLog);
                        }

                        PktPacket packet = entry.getPacket();
                        PktPacket pendingPacket = pendingPackets.get(logName);
                        if (pendingPacket != null) {

                            /*
                             * update transaction boundaries
                             * 
                             * with filtering the packets denoting the start and end of a commit
                             * may be removed. We don't want there to be incomplete transactions
                             * midstream. 
                             */
                            long currentTransactionId = packet.getHeader().getODSSubheader().getTransactionId();
                            if (currentTransactionId != pendingPacket.getHeader().getODSSubheader().getTransactionId()) {
                                packet.getHeader().getODSSubheader().setFlagCommitStart(true);
                                pendingPacket.getHeader().getODSSubheader().setFlagCommitEnd(true);
                            }

                            // clear persister metadata (due to filtering it may not be valid)
                            pendingPacket.getHeader().getODSSubheader().copyPersisterMetadataFrom(null);

                            // write the packet
                            targetLog.writeCommitEntry(pendingPacket, false);

                            // dispose of the packet
                            pendingPacket.dispose();
                        }

                        packet.acquire();
                        pendingPackets.put(logName, packet);
                        entry.dispose();

                        recordsWritten++;
                        if (recordsWritten > 0 && (recordsWritten - 1) % 10000 == 0) {
                            console().info("..." + recordsWritten + " written");
                        }

                    }
                    else {
                        break;
                    }
                }

                /*
                 * write out final pending packets
                 * 
                 * note that we don't mark the pending packet as commit end explicitly
                 * ... the transaction will only be committed if the it was already a
                 * commit end. 
                 */
                for (String pendingLogName : pendingPackets.keySet()) {

                    RogLog targetLog = rewrittenLogs.get(pendingLogName);
                    PktPacket pendingPacket = pendingPackets.get(pendingLogName);
                    if (pendingPacket != null) {

                        // update transaction boundaries

                        // clear persister metadata (due to filtering it may not be valid)
                        pendingPacket.getHeader().getODSSubheader().copyPersisterMetadataFrom(null);

                        // write the packet
                        targetLog.writeCommitEntry(pendingPacket, false);

                        // dispose of the packet
                        pendingPacket.dispose();

                        console().info("..." + recordsWritten + " were rewritten to '" + pendingLogName + "'");
                    }
                }

            }
            finally {
                for (RogLog log : rewrittenLogs.values()) {
                    console().info("Flushing and closing rewritten log '" + log.getLogFile());
                    console().info("  " + log.getStats().toString());
                    log.flush(true);
                    log.close();
                }
            }
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public void interrupt(final Thread commandThread) {
            interrupted = true;
        }

    }

    /*
     * 'close' command handler.
     */
    @AnnotatedCommand.Command(keywords = "close", description = "Close open logs and queries")
    final public class Close extends AnnotatedCommand {
        @Argument(position = 1, name = "target", required = false, description = "Optionally specifies the name of a log or directory to close.")
        String target;

        @Override
        final public void execute() throws Exception {
            boolean closed = false;
            if (target == null) {
                if (reader != null) {
                    reader.close();
                    reader = null;
                    closed = true;
                }

                if (queryEngine != null) {
                    // closing the query engine should close the logs.
                    console().info("Closing query engine...");
                    closeQueryResult();
                    try {
                        queryEngine.close();
                        console().info("Query engine closed.");
                    }
                    catch (Exception e) {
                        console().error("Error closing query engine: " + e.getMessage(), e);
                    }

                    queryEngine = null;
                    closed = true;
                }

                for (RogLog log : openLogs.values()) {
                    console().info("Closing " + log.getName());
                    try {
                        log.close();
                    }
                    catch (Exception e) {
                        error("Error closing " + log.getName() + ": " + e.getMessage(), e);
                    }
                    closed = true;
                }
                openLogs.clear();

                if (!closed) {
                    console().info("No log to close.");
                }

                updatePrompt();
            }
            else {
                if (!openLogs.isEmpty()) {
                    console().info("Closing logs...");
                }
                if (!closeLogOrDirectory(new File(UtlFile.expandPath(target)))) {
                    listOpen();
                }
                else {
                    if (!openLogs.isEmpty()) {
                        console().info("Logs closed");
                    }
                }

                updatePrompt();

                closeQuery();

            }

        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public boolean interruptable() {
            return false;
        }
    }

    /*
     * 'compact' command handler.
     */
    @AnnotatedCommand.Command(keywords = { "logmetadata" },
                              description = "Shows log metadata for the given transaction log", hidden = true)
    final public class LogMetadata extends AnnotatedCommand {
        @Option(shortForm = 'u', longForm = "update", required = false, defaultValue = "false", description = "Instructs command to update a property ")
        boolean update;

        @Option(shortForm = 'l', longForm = "list", required = false, defaultValue = "true", description = "Lists metadata for the given log")
        boolean list;

        @Argument(position = 1, name = "logName", required = true, description = "The path to the log metata file.")
        File metadataFile;

        @RemainingArgs(name = "updates", required = false, description = "key=value pairs used to update metadata properties.")
        String[] updateValues;

        /* (non-Javadoc)
         * @see com.neeve.tools.interactive.commands.AnnotatedCommand#execute()
         */
        @Override
        public void execute() throws Exception {
            if (!metadataFile.exists()) {
                throw new IllegalArgumentException(metadataFile.getAbsolutePath() + " does not exist");
            }

            if (metadataFile.isDirectory()) {
                throw new IllegalArgumentException(metadataFile.getAbsolutePath() + " is not a file");
            }

            if (!metadataFile.getName().endsWith(".metadata")) {
                throw new IllegalArgumentException(metadataFile.getAbsolutePath() + " does not have a .metadata extension");
            }

            RogLogMetadata metadata = new RogLogMetadata(metadataFile.getParentFile(), metadataFile.getName().substring(0, metadataFile.getName().length() - 9));
            metadata.open();
            try {
                if (list) {
                    console().info(metadata.toString());
                }

                if (update) {
                    if (updateValues == null || updateValues.length == 0) {
                        throw new IllegalArgumentException("key=value pairs must be supplied when the update flag is set");
                    }

                    for (int i = 0; i < updateValues.length; i++) {
                        String[] kv = updateValues[i].split("=");
                        UtlReflection.setNonNestedProperty(metadata, kv[0], kv[1]);
                    }
                }
            }
            finally {
                metadata.close();
            }

        }
    }

    /*
     * 'compact' command handler.
     */
    @AnnotatedCommand.Command(keywords = { "compact" },
                              description = "Compacts a state replication log. (browse mode only)", hidden = true)
    final public class Compact extends AnnotatedCommand {
        @Override
        final public void execute() throws Exception {
            assertBrowseMode();
            assertOpen();
            RogLog log = reader.log();

            if (log.getLogFile().getName().endsWith("in.log")) {
                throw new Exception("Compaction not supported for inbound message logs");
            }

            if (log.getLogFile().getName().endsWith("out.log")) {
                throw new Exception("Compaction not supported for outbound message logs");
            }

            // if the query engine is open we need to retrieve the repository and close 
            // it to let go of file locks
            if (queryEngine != null) {
                closeQuery();
                for (QueryRepository<Long, RogLogReader.Entry> repo : queryEngine.getRepositories()) {
                    if (repo instanceof RogLogQueryRepository && repo.getName() == log.getName()) {
                        RogLogQueryRepository logRepo = (RogLogQueryRepository)repo;
                        if (logRepo.getLog() == log) {
                            logRepo.close();
                        }
                    }
                }
                queryEngine.removeRepository(getLogNameAlias(log));
            }

            // compact
            RogLogCompactor compactor = log.getCompactor();

            // compact
            compactor.compact();
            compactor.waitForCompactionToComplete();

            // recreate the reader (on new log)
            if (reader != null) {
                reader.close();
            }
            reader = log.createReader();

            // reregister with the query engine
            if (queryEngine != null) {
                RogLogQueryRepository repo = log.asRepository();
                repo.open();
                queryEngine.addRepository(repo, getLogNameAlias(log));
            }

            // sleep to let compactor console output flush
            Thread.sleep(200);

            //update the prompt to reflect any name change
            updatePrompt();
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public boolean interruptable() {
            return false;
        }
    }

    /*
     * 'cdc' command handler.
     */
    @AnnotatedCommand.Command(keywords = { "cdc" },
                              description = "Performs change data capture on the log. (browse mode only)", hidden = true)
    final public class Cdc extends AnnotatedCommand {
        @Option(shortForm = 'i', longForm = "initialSize", defaultValue = "10485760", required = false, description = "The initial size of the new file into which to compact")
        long initialLogSize;

        @Option(shortForm = 'w', longForm = "compactionWindow", defaultValue = "-1", required = false, description = "The compaction window size to use. A value <=0 indicates that the default windows size should be used.")
        long compactionWindowSize = -1;

        @Override
        final public void execute() throws Exception {
            assertBrowseMode();
            assertOpen();
            RogLog log = reader.log();

            if (log.getLogFile().getName().endsWith("in.log")) {
                throw new Exception("Compaction not supported for inbound message logs");
            }

            if (log.getLogFile().getName().endsWith("out.log")) {
                throw new Exception("Compaction not supported for outbound message logs");
            }

            // if the query engine is open we need to retrieve the repository and close 
            // it to let go of file locks
            if (queryEngine != null) {
                closeQuery();
                for (QueryRepository<Long, RogLogReader.Entry> repo : queryEngine.getRepositories()) {
                    if (repo instanceof RogLogQueryRepository && repo.getName() == log.getName()) {
                        RogLogQueryRepository logRepo = (RogLogQueryRepository)repo;
                        if (logRepo.getLog() == log) {
                            logRepo.close();
                        }
                    }
                }
                queryEngine.removeRepository(getLogNameAlias(log));
            }

            // cdc
            RogLogCdcProcessor cdcProcessor = log.createCdcProcessor(new NoOpCdcHandler());
            cdcProcessor.run();
            cdcProcessor.close();

            // recreate the reader (on new log)
            if (reader != null) {
                reader.close();
            }
            reader = log.createReader();

            // reregister with the query engine
            if (queryEngine != null) {
                RogLogQueryRepository repo = log.asRepository();
                repo.open();
                queryEngine.addRepository(repo, getLogNameAlias(log));
            }

            // sleep to let compactor console output flush
            Thread.sleep(200);

            //update the prompt to reflect any name change
            updatePrompt();
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public boolean interruptable() {
            return false;
        }
    }

    private final class NoOpCdcHandler implements IRogChangeDataCaptureHandler {

        /* (non-Javadoc)
         * @see com.neeve.rog.IRogChangeDataCaptureHandler#onLogStart(int)
         */
        @Override
        public void onLogStart(int logNumber) {
            interactiveTool.info("[CDC HANDLER] Starting CDC on Log #" + logNumber + "...");
        }

        /* (non-Javadoc)
         * @see com.neeve.rog.IRogChangeDataCaptureHandler#onLogComplete(int, com.neeve.rog.IRogChangeDataCaptureHandler.LogCompletionReason, java.lang.Throwable)
         */
        @Override
        public void onLogComplete(int logNumber, LogCompletionReason reason, Throwable errorCause) {
            if (errorCause != null) {
                interactiveTool.error("[CDC HANDLER] Completed CDC on Log #" + logNumber + "(" + reason + ") - " + errorCause.getMessage(), errorCause);
            }
            else {
                interactiveTool.info("[CDC HANDLER] Completed CDC on Log #" + logNumber + "(" + reason + ")");
            }
        }

        /* (non-Javadoc)
         * @see com.neeve.rog.IRogChangeDataCaptureHandler#onCheckpointStart(long)
         */
        @Override
        public void onCheckpointStart(long checkpointVersion) {
            interactiveTool.info("[CDC HANDLER] On Checkpoint #" + checkpointVersion + " start.");
        }

        /* (non-Javadoc)
         * @see com.neeve.rog.IRogChangeDataCaptureHandler#onCheckpointComplete(long)
         */
        @Override
        public boolean onCheckpointComplete(long checkpointVersion) {
            interactiveTool.info("[CDC HANDLER] On Checkpoint #" + checkpointVersion + " complete.");
            return true;
        }

        /* (non-Javadoc)
         * @see com.neeve.rog.IRogChangeDataCaptureHandler#onWait()
         */
        @Override
        public void onWait() {
            interactiveTool.info("[CDC HANDLER] On Wait.");
        }

        /* (non-Javadoc)
         * @see com.neeve.rog.IRogChangeDataCaptureHandler#handleChange(com.eaio.uuid.UUID, com.neeve.rog.IRogChangeDataCaptureHandler.ChangeType, java.util.List)
         */
        @Override
        public boolean handleChange(UUID objectId, ChangeType changeType, List<IRogNode> entries) {
            interactiveTool.info("[CDC HANDLER] on change [" + (entries.size() > 0 ? entries.get(0).getClass().getSimpleName() : "???") + "(id=" + objectId + "), ctype " + changeType + ", entries=" + entries.size() + "]");
            return true;
        }
    }

    /*
     * 'compare' command handler.
     */
    @AnnotatedCommand.Command(keywords = "compare", description = "Compares two logs")
    final public class Compare extends AnnotatedCommand {

        @Option(shortForm = 'v', longForm = "verbose", defaultValue = "false", required = false, description = "Whether to dump comparison information for debugging purposes")
        boolean verbose;

        @Option(shortForm = 'c', longForm = "csv", defaultValue = "false", description = "indicates that csv mode should be used when displaying diffs in the console, otherwise a tabular format will be used")
        boolean csv;

        @Option(shortForm = 'm', longForm = "metadata", defaultValue = "false", required = false, description = "Indicates that platform metadata should be considered as well")
        boolean metadata;

        @Option(shortForm = 's', longForm = "state", defaultValue = "false", required = false, description = "If the two logs are state replication recovery logs, this flag will compare their rebuilt state")
        boolean state;

        @Option(shortForm = 'l', longForm = "limit", defaultValue = "5", required = false, description = "The number of divergent entries to display or dump once the first divergence is detected")
        int limit;

        @Argument(position = 1, required = true, name = "log1Name", description = "The name of log to compare relative to the current working directory (no .log suffix)")
        File log1;

        @Argument(position = 2, required = true, name = "log2Name", description = "The name of second log to compare relative to the current working directory (no .log suffix).")
        File log2;

        @Argument(position = 3, required = false, name = "outputFile", description = "The name of a file to which to dump diff results. A suffix of .csv will yield csv output, .html will yield html, otherwise a textual tabular format is used.")
        File dumpFile;

        @Override
        final public void execute() throws Exception {

            openLog(log2, false, "r", null);
            openLog(log1, false, "r", null);

            switchTo(log1);

            try {
                RogLog first = openLogs.get(mainLogFile(log1));
                RogLog second = openLogs.get(mainLogFile(log2));

                Format format = Format.TABULAR;
                final BufferedWriter bw;
                if (dumpFile != null) {
                    if (dumpFile.exists()) {
                        throw new Exception("File already exists");
                    }
                    if (dumpFile.getName().endsWith(".html")) {
                        format = Format.HTML;
                    }
                    else if (dumpFile.getName().endsWith(".csv")) {
                        format = Format.CSV;
                    }

                    bw = new BufferedWriter(new FileWriter(dumpFile));
                }
                else {
                    bw = new BufferedWriter(new PrintWriter(console().out()));
                }

                if (csv) {
                    format = Format.CSV;
                }

                boolean divergent = false;
                if (state) {
                    divergent = !RogLogUtil.compareState(first, second, RogLogUtil.loadComparisonFilter(), bw, metadata, verbose, format);
                }
                else {
                    divergent = !RogLogUtil.compareEntries(first, second, RogLogUtil.loadComparisonFilter(), bw, limit, metadata, verbose, format);
                }

                //Console feedback if we're dumping to a file:
                if (dumpFile != null) {
                    if (!divergent) {
                        console().info("Logs are not divergent.");
                    }
                    else {
                        console().info("Logs are divergent.");
                    }
                }

                if (dumpFile != null) {
                    bw.close();
                }
            }
            catch (Exception e) {
                throw new Exception("Error comparing logs:" + e.getMessage(), e);
            }
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public boolean interruptable() {
            return false;
        }
    }

    /*
     * 'create' command handler.
     */
    @AnnotatedCommand.Command(keywords = { "create", "CREATE" }, description = "Creates log indexes.")
    final public class Create extends AnnotatedCommand {
        @RemainingArgs(name = "indexArgs", required = true, description = "<optional:UNIQUE> INDEX <name> ON <log_name|logs>(<indexField>) indexField is specified the same as it would in a select or where clause.")
        String remainingArgs;

        @Override
        final public void execute() throws Exception {
            assertOpen();
            if (mode != Mode.QUERY) {
                switchTo("query");
            }
            queryEngine.executeStatement("CREATE " + remainingArgs);
        }
    }

    /*
     * 'drop' command handler.
     */
    @AnnotatedCommand.Command(keywords = { "drop", "DROP" }, description = "Drops a log index")
    final public class Drop extends AnnotatedCommand {
        @RemainingArgs(required = true, name = "name", description = "the drop statement 'DROP INDEX <indexName>', Use 'list' indexes to see existing indexes")
        String remainingArgs;

        @Override
        final public void execute() throws Exception {
            assertQueryMode();

            queryEngine.executeStatement("DROP " + remainingArgs);
        }
    }

    /*
     * 'select' command handler.
     * 
     * Note that we preserve arguments quotes during parsing so that in a statement like
     * 
     * SELECT * from logs WHERE simpleClassName = 'Customer'
     * 
     * ... the single quotes around Customer are preserved. 
     */
    @AnnotatedCommand.Command(keywords = { "SELECT", "select" }, parseOptions = false, preserveArgumentQuotes = true, description = "Issues a select statement against one or more open logs.")
    final public class Select extends AnnotatedCommand {
        @RemainingArgs(name = "query", required = true, description = "<fields> FROM <logs> WHERE <conditions>. The keyword 'logs' in the FROM clause selects all open logs")
        String remainingArgs;

        @Override
        final public void execute() throws Exception {
            assertOpen();
            closeQueryResult();
            if (mode != Mode.QUERY) {
                switchTo("query");
            }

            Stopwatch timer = Stopwatch.createStarted();
            queryResults = queryEngine.execute(queryEngine.createQuery("SELECT " + remainingArgs));

            int estimate = queryResults.getEstimatedCount();

            boolean showPreview = interactiveTool.isInteractive();
            boolean showQueryPlan = interactiveTool.isInteractive() && TltCommand.this.showQueryPlan;

            if (showQueryPlan) {
                queryResults.describePlan(interactiveTool.out());
            }

            console().info("Query plan prepared in " + timer.toString() + ".");

            if (queryResults.isFullScan()) {
                console().info("  No suitable indexes found for query, full scan will be performed to serve results.");
            }
            else {
                //TODO would be nice if we could indicate if the result is approximate or not. 
                console().info("  Approximately " + estimate + " entries need to be inspected to show results.");
            }

            if (selectPreviewCount > 0 && showPreview) {
                console().info("Result Preview:");
                int i = 0;
                while (queryResults.next() && i < selectPreviewCount) {
                    writeQueryResult(console().out(), i == 0, Format.TABULAR);
                    i++;
                }
                if (i == 0) {
                    console().info("...no results.");
                }

                //Rewind:
                if (i > 0) {
                    queryResults.beforeFirst();
                    console().info("Results rewound.");
                }
            }

            console().info("Use 'next', 'next <count>', 'dump' or 'count' commands to display results");
        }
    }

    /*
     * 'list' command handler.
     */
    @AnnotatedCommand.Command(keywords = { "LIST", "list" }, description = "Lists objects.")
    final public class ListCommand extends AnnotatedCommand {
        @Option(shortForm = 'v', longForm = "verbose", defaultValue = "false", required = false, description = "Whether to list extra details if avaiable")
        boolean verbose;

        @Argument(position = 1, name = "type", required = true, validOptions = { "indexes", "types", "logs", "open", "schema", "commands" },
                  description = "What to list." +
                          " A value of 'indexes' lists the indexes on open logs." +
                          " A value of 'types' lists the types available for query." +
                          " A value of 'logs' or 'open' lists the currently open logs.")
        String type;

        @Argument(position = 2, name = "filter", required = false, description = "An optional, case sensitive filter on list")
        String filter;

        private volatile boolean interrupted = false;

        @Override
        final public void execute() throws Exception {
            interrupted = false;
            if (type.equalsIgnoreCase("indexes")) {
                assertOpen();
                switchTo("query");
                Set<IdxField<RogLogReader.Entry, ?>> indexedFields = queryEngine.getIndexedFields();
                if (!indexedFields.isEmpty()) {
                    for (IdxField<RogLogReader.Entry, ?> field : indexedFields) {
                        if (interrupted) {
                            break;
                        }

                        String line = (verbose ? field.getCanonicalName() : field.getName()) + ": path=" + field.getFieldPath();
                        if (verbose) {
                            line += ", fieldType=" + field.getFieldType().getSimpleName() + ", recordType=" + field.getRecordType().getName();
                        }
                        if (filter != null && line.indexOf(filter) == -1) {
                            continue;
                        }
                        console().info(line);

                        if (verbose) {
                            for (QueryRepository<Long, RogLogReader.Entry> repo : queryEngine.getRepositories()) {
                                if (repo.getIndex(field) == null) {
                                    console().info("  " + repo.getName() + ": NO INDEX");
                                    continue;
                                }
                                IdxIndex<?, ?> index = repo.getIndex(field);
                                IdxIndex.Stats<?> stats = repo.getIndex(field).getStats();
                                console().info("  " + repo.getName() + "indexer : " + index + ":");
                                console().info("     live        : " + stats.isLive());
                                console().info("     current     : " + ((QueryIndexableRepository<?, ?>)repo).isLiveIndexingUpToDate());
                                console().info("     commitSeqNo : " + stats.getCommitSequenceNumber());
                                console().info("     cardinality : " + stats.getCardinality());
                                console().info("     keys        : " + stats.getKeyCardinality());
                                console().info("     lowKey      : " + stats.getLowKey());
                                console().info("     highKey     : " + stats.getHighKey());
                            }
                        }
                    }
                }
                else {
                    console().info("No indexes found for the currently opened log(s). Use 'create index' command to create one");
                }
            }
            else if (type.equalsIgnoreCase("types")) {
                assertOpen();
                switchTo("query");
                for (Class<?> type : ((RogLogQueryEngineImpl)queryEngine).getTypes()) {
                    if (interrupted) {
                        break;
                    }
                    // skip interfaces for now
                    if (type.isInterface()) {
                        continue;
                    }

                    final String line = type.getCanonicalName().replace(".", "/");
                    if (filter != null && line.indexOf(filter) == -1) {
                        continue;
                    }
                    console().info(line);
                }
            }
            else if (type.equalsIgnoreCase("logs") || type.equalsIgnoreCase("open")) {
                listOpen(filter, verbose);
            }
            else if (type.equalsIgnoreCase("schema")) {
                assertOpen();
                switchTo("query");
                HashSet<String> schemas = new HashSet<String>();
                for (Class<?> type : ((RogLogQueryEngineImpl)queryEngine).getTypes()) {
                    if (interrupted) {
                        break;
                    }
                    // skip interfaces for now
                    if (type.isInterface()) {
                        continue;
                    }

                    final String line = type.getPackage().getName();
                    if (filter != null && line.indexOf(filter) == -1) {
                        continue;
                    }
                    schemas.add(type.getPackage().getName());
                }

                List<String> schemaList = new ArrayList<String>(schemas);
                Collections.sort(schemaList);
                for (String schema : schemaList) {
                    console().info(schema);
                }
            }
            else if (type.equalsIgnoreCase("commands")) {
                interactiveTool.listCommands(filter);
            }
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public void interrupt(final Thread commandThread) {
            interrupted = true;
        }
    }

    /*
     * 'describe' command handler.
     */
    @AnnotatedCommand.Command(keywords = { "DESCRIBE", "describe" }, description = "Describes the fields for a given object.")
    final public class Describe extends AnnotatedCommand {
        @Argument(position = 1, name = "type", required = true, description = "The name of the type to describe, must be fully qualified e.g. com.mycompany.MyClass, if ambiguous")
        String type;

        @Override
        final public void execute() throws Exception {
            assertOpen();
            switchTo("query");
            RogLogQueryFieldResolver resolver = (RogLogQueryFieldResolver)((RogLogQueryEngineImpl)queryEngine).getQueryFieldResolver(RogLogReader.Entry.class);
            try {
                StringBuffer desc = new StringBuffer();
                resolver.describeFields(queryEngine.getField(type), desc);
                console().info(desc.toString());
            }
            catch (QueryException qe) {
                console().error("Error resolving type '" + type + "' " + qe.getMessage(), qe);
            }

        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public boolean interruptable() {
            return false;
        }
    }

    /*
     * 'schema' command handler.
     */
    @AnnotatedCommand.Command(keywords = { "SCHEMA", "schema", "NAMESPACE", "namespace", "PACKAGE", "package" }, description = "Sets the default package for resolving ambiguous unqualified class references")
    final public class Schema extends AnnotatedCommand {
        @Option(shortForm = 'x', longForm = "clear", defaultValue = "false", description = "Indicates that current default schema should be cleared")
        boolean clear;

        @Argument(position = 1, name = "packageName", required = false, description = "A package name e.g. com.foo or com/foo, required unless -x is specified")
        String packageName;

        @Override
        final public void execute() throws Exception {
            boolean changed = false;
            if (clear) {
                if (defaultPackage != null) {
                    changed = true;
                }
                defaultPackage = null;
            }
            else if (packageName != null && packageName.length() > 0) {
                Package newPackage = Package.getPackage(packageName.replace('/', '.'));
                if (newPackage == null) {
                    throw new IllegalArgumentException("Package not found: " + packageName);
                }
                else {
                    changed = defaultPackage == null || !defaultPackage.equals(newPackage);
                    defaultPackage = newPackage;
                }
            }

            if (queryEngine != null && changed) {
                queryEngine.setDefaultPackage(defaultPackage);
            }

            if (changed) {
                if (defaultPackage != null) {
                    console().info("Default package set to " + defaultPackage.getName());
                }
                else {
                    console().info("Default package cleared");
                }
            }
            else {
                if (defaultPackage != null) {
                    console().info("Default package is " + defaultPackage.getName());
                }
                else {
                    console().info("No default package set");
                }
            }
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public boolean interruptable() {
            return false;
        }
    }

    /*
     * 'count' command handler.
     */
    @AnnotatedCommand.Command(keywords = { "COUNT", "count" }, description = "Get the number of matching results in a query")
    final public class Count extends AnnotatedCommand {

        @Override
        final public void execute() throws Exception {
            assertQueryMode();
            assertResultSet();
            console().info("" + queryResults.getCount());
        }
    }

    /*
     * 'archive' command handler.
     */
    @AnnotatedCommand.Command(keywords = "archive", description = "Creates an executable jar archive of transaction logs")
    final public class Archive extends AnnotatedCommand {
        @Option(shortForm = 'm', longForm = "includeMessageLogs", defaultValue = "false", required = false, description = "flag indicating that corresponding inbound and outbound message logs should also be included in the archive")
        private boolean includeMessageLogs;

        @Option(shortForm = 's', longForm = "includePerTransactionStatsLog", defaultValue = "false", required = false, description = "flag indicating that corresponding transactionstats logs should also be included in the archive")
        private boolean includePerTransactionStats;

        @Option(shortForm = 'n', longForm = "name", required = false, description = "The name to use for the archive. If ommitted and a single log is specified its name will be used, otherwise 'archive' will be used.")
        private String name;

        @Option(shortForm = 'a', longForm = "additional", required = false,
                description = "A path delimited set of <archive-target-dir>=<source-fileOrDir-path> pairs" +
                        " indicating additional files or directories to include in the archive. Directories" +
                        " are copied into the archive recursively.\n           " +
                        " Examples\n           " +
                        " -a .=conf:logs=rdat/xserver.log:logs=/root/logs/mylog.log:.=errors.txt\n           " +
                        " Would result in\n           " +
                        " - conf being recursively copied to conf in the archive\n           " +
                        " - rdat/xserver.log being copied to logs/xserver.log\n           " +
                        " - /root/logs/mylog.log being copied to logs/mylog.log\n           " +
                        " - errors.txt being copied to errors.txt in the archive\n           ")
        private String additionalContent;

        @Argument(position = 1, name = "log", required = true, description = "The path to the log that should be archived")
        File log;

        @RemainingArgs(name = "additionalLogs", required = false, description = "Additional logs to add to the archive")
        String[] additionalLogs;

        @Override
        final public void execute() throws Exception {
            if (log.isDirectory()) {
                throw new Exception("Can't archive a directory, a log file must be specified");
            }

            // the list of logs to include for archive
            TreeMap<String, RogLog> included = new TreeMap<String, RogLog>();

            // track which open logs were closed for archiving
            ArrayList<File> toReopen = new ArrayList<File>();

            // open the specified log for archiving
            openLogsForArchive(log, included, toReopen);

            // plus any additional
            for (String additional : additionalLogs) {
                File additionalFile = new File(UtlFile.expandPath(additional));
                if (additionalFile.isDirectory()) {
                    console().error("Can't archive a directory, a log file must be specified, " + additionalFile + " will be ignored");
                    continue;
                }
                openLogsForArchive(additionalFile, included, toReopen);
            }

            // initialize the archive name
            if (Strings.isNullOrEmpty(name)) {
                // if there are multiple logs specified defatul to 'archive'
                if (additionalLogs != null && additionalLogs.length > 0) {
                    name = "archive";
                }
                else {
                    // otherwise use the single log name for archive name
                    name = included.firstEntry().getValue().getName();
                }
            }

            // parse additional archive contents
            List<String> additionalContentList = null;
            if (additionalContent != null) {
                additionalContentList = new ArrayList<String>();
                StringTokenizer contentTok = new StringTokenizer(additionalContent, File.pathSeparator);
                while (contentTok.hasMoreTokens()) {
                    additionalContentList.add(contentTok.nextToken());
                }
            }

            try {
                File archive = RogLogUtil.createExecutableArchive(new File(workingDir, "archives"), name, console().out(), null, additionalContentList, included.values().toArray(new RogLog[] {}));
                console().info("Successfully created archive: " + UtlFile.relativePath(workingDir, archive) + " (" + UtlFile.readableFileSize(archive.length()) + ")");
            }
            finally {
                for (RogLog archived : included.values()) {
                    archived.close();
                }
            }
        }

        /**
         * Opens the log in a state ready for archive. This will also open inbound/outbound logs 
         * if requested. 
         *  
         * @param log The log file. 
         *  
         * @param logs The opened logs are added to this list. 
         *  
         * @param toReopen If a log was closed to prepare for archiving, it will be added to this list
         *  
         * @throws Exception If there was an error opening the log. 
         */
        private void openLogsForArchive(File log, Map<String, RogLog> logs, List<File> toReopen) throws Exception {
            // open the main log
            log = mainLogFile(log);
            RogLog main = openLogForArchive(log, toReopen);
            if (main == null) {
                throw new Exception("Log file " + log + " does not exist");
            }
            else {
                logs.put(main.getLogFile().getCanonicalPath(), main);
            }

            if (includeMessageLogs) {
                RogLog inbound = openLogForArchive(new File(main.getLogFile().getParent(), main.getName() + ".in.log"), toReopen);
                if (inbound != null) {
                    logs.put(inbound.getLogFile().getCanonicalPath(), inbound);
                }

                RogLog outbound = openLogForArchive(new File(main.getLogFile().getParent(), main.getName() + ".out.log"), toReopen);
                if (outbound != null) {
                    logs.put(outbound.getLogFile().getCanonicalPath(), outbound);
                }
            }

            if (includePerTransactionStats) {
                RogLog txnstats = openLogForArchive(new File(main.getLogFile().getParent(), main.getName() + ".txnstats.log"), toReopen);
                if (txnstats != null) {
                    logs.put(txnstats.getLogFile().getCanonicalPath(), txnstats);
                }
            }
        }

        /**
         * Opens a log for archive if it exists.
         * 
         * @param log The log to open.
         *  
         * @param toReopen If a log was closed to prepare for archiving, it will be added to this list
         *  
         * @return The log if the log exists, null otherwise. 
         *  
         * @throws Exception If there is an error opening the log.
         */
        final private RogLog openLogForArchive(File log, List<File> toReopen) throws Exception {
            log = mainLogFile(log);
            File metadata = new File(log.getParentFile(), log.getName().substring(0, log.getName().length() - 4) + ".metadata");

            if (!metadata.exists()) {
                return null;
            }

            // logs need to be closed for archive
            if (openLogs.containsKey(log)) {
                closeLog(log);
                toReopen.add(log);
            }

            return createLog(log, false, "r", null);
        }

        /**
         * Called to interrupt a command.
         */
        @Override
        public boolean interruptable() {
            return false;
        }
    }

    /*
     * Config property
     */
    @ConfigProperty(name = "lazyDeserialization",
                    validOptions = { "lazy", "eager" },
                    defaultValue = "lazy",
                    description = "Whether or not objects should be be lazily deserialized on demand.")
    private DeserializationPolicy deserializationPolicy;

    @ConfigProperty(name = "displayMetadata",
                    defaultValue = "On",
                    validOptions = { "On", "Off", "Only" },
                    description = "Set whether metadata should be displayed by commands that dump entries in 'detailed' mode.\n " +
                            "-'On' indicates that both metadata and the object should be displayed.\n " +
                            "-'Off' indicates that only the object should be displayed.\n " +
                            "-'Only' indicates that only the metadata should be displayed (no object).")
    private MetadataPolicy metadataDisplay;

    @ConfigProperty(name = "filterUnsetFields",
                    defaultValue = "Off",
                    validOptions = { "on", "off", "true", "false" },
                    parser = InteractiveTool.BooleanPropertyParser.class,
                    description = "Sets whether to display values returned for unset fields for fields that expose a hasXXX method")
    private boolean filterUnsetFields = false;

    @ConfigProperty(name = "jsonStyle",
                    defaultValue = "Default",
                    validOptions = { "Default", "PrettyPrint", "Minimal", "SingleLine", "Custom" },
                    description = "Sets the pretty print style to use for json output.\n" +
                            "-Default: Jackson default pretty printer\n" +
                            "-PrettyPrint: Jackson default pretty printer (same as default)\n" +
                            "-Minimal: A minimal single line printer\n" +
                            "-SingleLine: Single line with space after colon between field name and value\n" +
                            "-Custom: Examines the runtime property nv.json.customPrettyPrinter to find " +
                            "the classname of a pretty printer that implements:\n   " +
                            "com.fasterxml.jackson.core.PrettyPrinter")
    private JsonPrettyPrintStyle jsonPrettyPrintStyle = JsonPrettyPrintStyle.Default;

    @ConfigProperty(name = "autoindexing",
                    defaultValue = "off",
                    validOptions = { "off", "on", "true", "false" },
                    parser = InteractiveTool.BooleanPropertyParser.class,
                    description = "Sets whether to automatically create indexes based on queries.\n " +
                            "'on' Enables auto indexing \n" +
                            "'off' disables auto indexing. \n\n" +
                            "Enabling auto indexing will create indexes based on fields referenced in " +
                            "a query to improve response time of subsequent follow up queries, but it " +
                            "can lead to progressively slower indexing of new entries if too many " +
                            "indexes are auto created, and can additionally slow down query results " +
                            "while the indexes are being built depending on the 'indexingPolicy'")
    private boolean autoindexing = false;

    @ConfigProperty(name = "autoindexLimit",
                    defaultValue = "5",
                    description = "The number of fields to be indexed via autoindexing when executing " +
                            "select statements. The limit considers fields from the WHERE, ORDER BY, " +
                            "GROUP BY and SELECT clauses of a query in that order.")
    private int autoindexLimit = 5;

    @ConfigProperty(name = "indexingPolicy",
                    defaultValue = "FLUSH_ON_CREATE",
                    validOptions = { "FLUSH_NEVER", "FLUSH_ON_CREATE", "FLUSH_ON_QUERY", "FLUSH_ON_USE" },
                    description = "Sets the policy for background indexing, if none specified the current value will be displayed\n" +
                            "-FLUSH_NEVER: If applicable index is still being built don't wait, instead, start with table scan\n" +
                            "-FLUSH_ON_USE: If an index is still being built, and can be used by a query, query will wait for index build to finish\n" +
                            "-FLUSH_ON_QUERY:' Flush all indexing prior to executing a SELECT statement\n" +
                            "-FLUSH_ON_CREATE: Synchronously build the index when it is created. 'create index' commands and " +
                            "'select' with autoindexing=on will wait for index build prior to executing the query.")
    private BackgroundIndexingPolicy backgroundIndexingPolicy = BackgroundIndexingPolicy.FLUSH_ON_CREATE;

    @ConfigProperty(name = "fullscanThreshold",
                    parser = FullscanThresholdParser.class,
                    description = "A value between 0 and 1 representing the minimum proportion of an index's keys that qualify the index to be used in a query")
    private Double fullscanThreshold = null;

    @ConfigProperty(name = "selectPreviewCount",
                    defaultValue = "0",
                    description = "A positive value indicating the default number of rows to preview after a select.")
    private int selectPreviewCount = 0;

    @ConfigProperty(name = "deleteExtractedArchivesDirOnExit",
                    defaultValue = "off",
                    validOptions = { "off", "on", "true", "false" },
                    parser = InteractiveTool.BooleanPropertyParser.class,
                    description = " Enabling this option will recursively delete the extracted-archives folder when exiting the tool " +
                            "including any previously files in the directory so care should be taken when using this option.")
    private boolean deleteExtractedArchivesDirOnExit = false;

    @ConfigProperty(name = "showQueryPlan",
                    defaultValue = "false",
                    validOptions = { "off", "on", "true", "false" },
                    parser = InteractiveTool.BooleanPropertyParser.class,
                    description = "Whether or not a query's plan is displayed after being issue.")
    private boolean showQueryPlan = false;

    @ConfigProperty(name = "maxInMemorySortCardinality",
                    defaultValue = "500",
                    description = "The size beyond which temporary data structures for sorting will become disk based.")
    private int maxInMemorySortCardinality = 500;

    @ConfigProperty(name = "timestampFormat",
                    defaultValue = "yyyy-MM-dd HH:mm:ss.SSS z",
                    description = "The default timestamp format to use when displaying results (use quotes when setting).")
    private String timestampFormat = "yyyy-MM-dd HH:mm:ss.SSS z";

    @ConfigProperty(name = "timestampTZ",
                    defaultValue = "DEFAULT",
                    description = "The timezone in which to display timestamps. A value of 'DEFAULT' uses the current system timezone")
    private String timestampTZ = TimeZone.getDefault().getID();

    @ConfigProperty(name = "logScavengePolicy",
                    defaultValue = "Disabled",
                    validOptions = { "Disabled", "Delete" },
                    description = "Whether or not cleanup of old log files that are no longer needed by CDC. For safety this value is defaulted to false.")
    private String logScavengePolicy = "Disabled";

    /*
     * Private members
     */
    private InteractiveTool interactiveTool;
    private Mode mode = Mode.BROWSE;
    private File workingDir;
    private RogLogReader reader;
    private Package defaultPackage = null;
    private RogLogQueryEngine queryEngine = null;
    private RogLogQueryResultSet queryResults = null;
    private HashMap<Format, String> resultHeaders = new HashMap<Format, String>();
    private HashMap<Format, String> resultRowFormats = new HashMap<Format, String>();
    private HashMap<Format, String> resultFooters = new HashMap<Format, String>();
    private File extractedArchivesDir = null;
    final private FileFilter logMetadataFilter = new FileFilter() {
        @Override
        public boolean accept(File pathname) {
            return pathname.isFile() && pathname.getName().endsWith(".metadata");
        }
    };
    private HashMap<File, RogLog> openLogs = new HashMap<File, RogLog>();

    final TltCommand init(File archiveExtractionDir) throws Exception {
        this.interactiveTool = new InteractiveTool("Transaction Log Tool", null, ManifestProductInfo.loadProductInfo("nvx-rumi-tools"));
        interactiveTool.registerAnnotatedCommands(this);
        interactiveTool.addCloseHook(new Runnable() {

            @Override
            public void run() {
                try {
                    new Close().execute();
                }
                catch (Exception e) {
                    throw new RuntimeException("Error closing logs: " + e.getMessage(), e);
                }
                finally {
                    if (deleteExtractedArchivesDirOnExit && TltCommand.this.extractedArchivesDir != null && TltCommand.this.extractedArchivesDir.exists()) {
                        interactiveTool.info("Deleting extracted archive directory '" + TltCommand.this.extractedArchivesDir + "'...");
                        try {
                            UtlFile.deleteDirectory(TltCommand.this.extractedArchivesDir);
                        }
                        catch (IOException e) {
                            interactiveTool.error("Error deleting extracted archive directory", e);
                        }
                    }
                }
            }
        });

        // Pre command loop tasks:
        interactiveTool.addPreCommandLoopTask(new Runnable() {

            @Override
            public void run() {
                if (!"disabled".equalsIgnoreCase(logScavengePolicy)) {
                    interactiveTool.warn("WARNING: logScavengePolicy is set to '" + logScavengePolicy + "'. Ensure any old log versions needed by CDC are backed up!");
                }
                if (!"disabled".equalsIgnoreCase(logScavengePolicy)) {
                    interactiveTool.warn("WARNING: Before operating on any production files, ensure they are first backed up!");
                }

                // extract and open any archived logs
                try {
                    List<File> embeddedFiles = RogLogUtil.extractArchivedLogs(interactiveTool.out(), TltCommand.this.extractedArchivesDir);
                    if (embeddedFiles != null && !embeddedFiles.isEmpty()) {
                        for (File embedded : embeddedFiles) {
                            try {
                                openLog(embedded, false, "r", null);
                            }
                            catch (Exception e) {
                                interactiveTool.error("Error opening embedded log: " + e.getMessage(), e);
                            }
                        }
                        switchTo(embeddedFiles.get(0));
                    }
                }
                catch (Exception e) {
                    interactiveTool.error("Error extracting embedded archive contents: " + e.getMessage(), e);
                }

                // run any init scripts:
                String initScripts = Config.getValue(PROP_TLT_INIT_SCRIPTS, null);
                if (initScripts != null) {
                    StringTokenizer tok = new StringTokenizer(initScripts, ",");
                    while (tok.hasMoreTokens()) {
                        File initScript = new File(tok.nextToken());
                        if (!initScript.exists()) {
                            interactiveTool.error("Init script not found " + initScript);
                        }

                        if (initScript.isDirectory()) {
                            interactiveTool.error("Init script not a file " + initScript);
                        }

                        BufferedReader reader = null;
                        try {
                            interactiveTool.info("Running init script " + initScript);

                            reader = new BufferedReader(new FileReader(initScript));
                            String line = reader.readLine();
                            while (line != null) {
                                interactiveTool.executeCommand(line, true);
                                line = reader.readLine();
                            }
                        }
                        catch (Exception e) {
                            interactiveTool.error("Init script not a file " + initScript);
                        }
                        finally {
                            if (reader != null) {
                                try {
                                    reader.close();
                                }
                                catch (IOException e) {
                                    interactiveTool.error("Init script not properly closed " + initScript, e);
                                }
                            }
                        }
                    }
                }
            }
        });

        workingDir = new File(System.getProperty("user.dir"));

        // register change handlers
        interactiveTool.addConfigPropertyChangeHandler("autoindexing", new PropertyChangeHandler() {

            @Override
            public void onPropertyChange(String property, Object value) {
                if (queryEngine != null) {
                    queryEngine.setAutoIndexing(autoindexing);
                }
            }

        });

        interactiveTool.addConfigPropertyChangeHandler("autoindexLimit", new PropertyChangeHandler() {

            @Override
            public void onPropertyChange(String property, Object value) {
                if (queryEngine != null) {
                    queryEngine.setAutoIndexLimit(autoindexLimit);
                }
            }

        });

        interactiveTool.addConfigPropertyChangeHandler("indexingPolicy", new PropertyChangeHandler() {

            @Override
            public void onPropertyChange(String property, Object value) {
                if (queryEngine != null) {
                    queryEngine.setBackgroundIndexingPolicy(backgroundIndexingPolicy);
                }
            }
        });

        interactiveTool.addConfigPropertyChangeHandler("fullscanThreshold", new PropertyChangeHandler() {

            @Override
            public void onPropertyChange(String property, Object value) {
                if (queryEngine != null) {
                    queryEngine.setFetchRatioThreshold(fullscanThreshold);
                }
            }
        });

        interactiveTool.addConfigPropertyChangeHandler("maxInMemorySortCardinality", new PropertyChangeHandler() {

            @Override
            public void onPropertyChange(String property, Object value) {
                if (queryEngine != null) {
                    queryEngine.setSortAreaInMemoryCardinality(maxInMemorySortCardinality);
                }

            }
        });

        interactiveTool.addConfigPropertyChangeHandler("lazyDeserialization", new PropertyChangeHandler() {

            @Override
            public void onPropertyChange(String property, Object value) {
                if (reader != null) {
                    reader.setLazyDeserialization(deserializationPolicy == DeserializationPolicy.Lazy);
                }
            }
        });

        UtlTime.setTimestampFormat(timestampFormat);
        interactiveTool.addConfigPropertyChangeHandler("timestampFormat", new PropertyChangeHandler() {

            @Override
            public void onPropertyChange(String property, Object value) {
                UtlTime.setTimestampFormat(value.toString());
            }
        });

        if ("DEFAULT".equals(timestampTZ)) {
            timestampTZ = TimeZone.getDefault().getID();
        }
        UtlTime.setTimestampFormatTimeZone(timestampTZ);
        interactiveTool.addConfigPropertyChangeHandler("timestampTZ", new PropertyChangeHandler() {

            @Override
            public void onPropertyChange(String property, Object value) {
                if ("DEFAULT".equals(value)) {
                    value = TimeZone.getDefault().getID();
                }
                UtlTime.setTimestampFormatTimeZone(value.toString());
            }
        });

        interactiveTool.addConfigPropertyChangeHandler("logScavengePolicy", new PropertyChangeHandler() {

            @Override
            public void onPropertyChange(String property, Object value) {
                if (!"disabled".equalsIgnoreCase(String.valueOf(value))) {
                    interactiveTool.info("...WARNING: log scavenge policy is set to '" + value + "'. Ensure any old log versions needed by CDC are backed up!");
                }
            }
        });

        if (archiveExtractionDir == null) {
            this.extractedArchivesDir = new File(workingDir, "extracted-archives");
        }
        else {
            this.extractedArchivesDir = new File(archiveExtractionDir, "extracted-archives");
        }

        return this;
    }

    /*
     * Main entry point
     */
    final private void run() throws Exception {
        Thread.currentThread().setName("X-TLT");
        interactiveTool.run();
    }

    /*
     * Main entry point for script file
     */
    final private void run(File script) throws Exception {
        Thread.currentThread().setName("X-TLT");
        interactiveTool.run(script);
    }

    final private void switchTo(File logfile) throws Exception {
        RogLog log = openLogs.get(mainLogFile(logfile));
        if (log != null) {
            if (reader == null || !reader.log().getLogFile().equals(log.getLogFile())) {
                if (reader != null) {
                    reader.close();
                    interactiveTool.info("Switching to '" + log.getName() + "' use 'switch <logName>' to change between browsed logs");
                }
                reader = log.createReader();
                reader.setLazyDeserialization(deserializationPolicy == DeserializationPolicy.Lazy);
                interactiveTool.info("opened with " + deserializationPolicy + " deserialization.");
            }
            mode = Mode.BROWSE;
        }
        else {
            interactiveTool.error("No log named '" + log + " is currently open");
            listOpen();
            return;
        }

        updatePrompt();
    }

    final private void switchTo(final String target) throws Exception {
        if (target.equalsIgnoreCase("query")) {
            if (mode != Mode.QUERY) {
                if (queryEngine == null) {
                    interactiveTool.info("Initializing query engine.");
                    queryEngine = RogLogFactory.createQueryEngine();

                    // add any open logs to the query engine:
                    if (openLogs != null) {
                        for (RogLog log : openLogs.values()) {
                            RogLogQueryRepository repo = log.asRepository();
                            repo.open();
                            queryEngine.addRepository(repo, getLogNameAlias(log));
                        }
                    }
                    queryEngine.setDefaultPackage(defaultPackage);
                    queryEngine.setAutoIndexing(autoindexing);
                    queryEngine.setAutoIndexLimit(autoindexLimit);
                    queryEngine.setBackgroundIndexingPolicy(backgroundIndexingPolicy);
                    if (fullscanThreshold != null) {
                        queryEngine.setFetchRatioThreshold(fullscanThreshold);
                    }
                }
                mode = Mode.QUERY;
            }
            else {
                return;
            }

            interactiveTool.info("Entered query mode. Use 'switch browse' to go back to log mode");
        }
        else if (target.equalsIgnoreCase("browse")) {
            if (mode != Mode.BROWSE) {
                mode = Mode.BROWSE;
            }
            else {
                return;
            }
        }
        else {
            //Test if this is a log file.
            switchTo(mainLogFile(new File(target)));
        }

        updatePrompt();
    }

    final private void updatePrompt() throws IOException {
        if (mode == Mode.QUERY) {
            interactiveTool.setPrompt("query");
        }
        else {
            if (reader == null) {
                interactiveTool.setPrompt(null);
            }
            else {
                interactiveTool.setPrompt(getRelativeLogAlias(reader.log()));
            }
        }
    }

    /**
     * Returns the main log file. Transaction log files that are compacted for 
     * checkpoints have are of the form <name>[.version].log. This method
     * method accepts Files that ommit the .log suffix or checkpoint version
     * and correct the file to be of the form <name>.log. 
     *  
     * <p>
     * To achieve this the following algorithm is used:
     * <ol>
     * <li>If the file does not end in .log, .log is appended to the file name. 
     * <li>If stripping the .log and appending .metadata to the name resolves to 
     * an existing file the the file from step 1 is returned
     * <li>Otherwise, if a metadata file exists by further stripping a .<checkpointVersion> from 
     * the file that file is returned. 
     * <li> Otherwise the the file arrived at in #2 is returned (assumes a file missing
     * its metadata
     * </ol>
     * 
     * @param template
     * @return An actual canonical .log File based on the template passed in.
     * @throws IOException
     */
    private File mainLogFile(File template) throws IOException {
        // normalize to .log format
        if (!template.getName().endsWith(".log")) {
            template = new File(template.getParentFile(), template.getName() + ".log");
        }

        // if there is a metadata file then this file was passed in main log form, return it
        if (new File(template.getParentFile(), template.getName().substring(0, template.getName().length() - ".log".length()) + ".metadata").exists()) {
            return template.getCanonicalFile();
        }

        // check for a versioned log file being passed in <name>.<version>.log
        String[] components = template.getName().split("\\.");
        if (components.length > 2) {
            try {
                Integer.parseInt(components[components.length - 1]);
                int versionSuffixPos = template.getName().substring(0, template.getName().length() - 4).lastIndexOf(".");
                File versionStripped = new File(template.getParentFile(), template.getName().substring(0, versionSuffixPos));
                if (new File(versionStripped.getParentFile(), versionStripped.getName() + ".metadata").exists()) {
                    return new File(versionStripped.getParentFile(), versionStripped.getName() + ".log");
                }
            }
            catch (NumberFormatException nfe) {
                //Not a versioned log. 
            }
        }

        return template.getCanonicalFile();
    }

    /**
     * Returns the log file path relative to the working directory with directory
     * separators normalized to '/'.
     * 
     * @param log The log for which to get the relative full
     * @return The log without file extension.
     * @throws IOException If there is a file system error getting the alias.
     */
    final private String getLogRelativePath(File log) throws IOException {
        return UtlFile.relativePath(workingDir, log).replace('\\', '/');
    }

    /**
     * Returns the log file's directory path relative to the working directory with directory
     * separators normalized to '/'.
     * 
     * @param log The log for which to get the relative full
     * @return The log relative path suffixed with '/' or empty string if the log is in the current directory
     * @throws IOException If there is a file system error getting the alias.
     */
    final private String getLogRelativeDirectory(RogLog log) throws IOException {
        String relPath = getLogRelativePath(log.getLogFile());
        return relPath.substring(0, relPath.length() - log.getLogFile().getName().length());
    }

    /**
     * Returns the log name without the ".log" or log number extension. 
     * 
     * @param log The log for which to get the alias.
     * @return The log without file extension.
     * @throws IOException If there is a file system error getting the alias.
     */
    final private String getRelativeLogAlias(RogLog log) throws IOException {
        String relativePath = getLogRelativeDirectory(log);
        if (relativePath.length() > 0) {
            return relativePath + getLogNameAlias(log);
        }
        else {
            return getLogNameAlias(log);
        }
    }

    /**
     * Returns the log name without the ".log" or log number extension. 
     * 
     * @param log The log for which to get the alias.
     * @return The log without file extension.
     * @throws IOException If there is a file system error getting the alias.
     */
    final private String getLogNameAlias(RogLog log) throws IOException {
        return log.getName();
    }

    final private void openLogOrDirectory(File file, boolean repair, String openMode, Properties properties) throws Exception {
        // if specified file is a directory, then open all logs in the directory
        if (file.exists() && file.isDirectory()) {
            // filter log metadata files
            for (File log : file.listFiles(logMetadataFilter)) {
                // open log deriving its name from the metadata file name
                File logFile = new File(file, log.getName().substring(0, log.getName().length() - ".metadata".length()));
                try {
                    openLog(logFile, repair, openMode, properties);
                }
                catch (Exception e) {
                    interactiveTool.error("Failed to open '" + logFile + "': " + e.getMessage(), e);
                }
            }
        }
        else {
            // otherwise, open/create the specified log
            openLog(file, repair, openMode, properties);
        }
    }

    final private void openLog(final File logfile, boolean repair, String openMode, Properties properties) throws Exception {
        File mainLogFile = mainLogFile(logfile).getCanonicalFile();
        String name = mainLogFile.getName();
        name = name.substring(0, name.length() - ".log".length());
        File metadata = new File(mainLogFile.getParentFile(), name + ".metadata");
        if (!metadata.exists()) {
            interactiveTool.error("Log doesn't exist, no metadata for '" + metadata + "' for '" + logfile + "'");
            return;
        }

        if (openLogs.containsKey(mainLogFile)) {
            interactiveTool.info("Log already opened: " + mainLogFile.getCanonicalPath());
            return;
        }

        interactiveTool.info("Opening log '" + getLogRelativePath(mainLogFile) + "'");

        final File factoriesFile = new File(mainLogFile.getParent(), name + ".factories");
        if (factoriesFile.exists()) {
            new Factories().doFactories(factoriesFile.getCanonicalPath());
        }
        else {
            interactiveTool.info("Factories file '" + factoriesFile + "' was not found.");
        }

        // Create properties if not passed in. 
        if (properties == null) {
            properties = new Properties();
        }

        // Set log scavenge policy:
        if (!properties.containsKey(RogLog.PROP_LOG_SCAVENGE_POLICY)) {
            properties.put(RogLog.PROP_LOG_SCAVENGE_POLICY, logScavengePolicy);
        }
        if (!"disabled".equalsIgnoreCase(properties.getProperty(RogLog.PROP_LOG_SCAVENGE_POLICY))) {
            interactiveTool.info("...log scavenge policy is set to '" + properties.getProperty(RogLog.PROP_LOG_SCAVENGE_POLICY) + "'.");
        }

        // Load metadata to make sure that we are not reseting CDC information:
        if (metadata.exists()) {
            RogLogMetadata logMetadata = new RogLogMetadata(metadata.getParentFile(), name);
            logMetadata.open();
            if (logMetadata.isCdcEnabled() && !properties.containsKey(RogLog.PROP_CDC_ENABLED)) {
                properties.put(RogLog.PROP_CDC_ENABLED, "" + logMetadata.isCdcEnabled());
                interactiveTool.info("...enabling CDC.");
            }
            logMetadata.close();
        }

        // open log
        final RogLog log = createLog(mainLogFile, repair, openMode, properties);
        log.open();
        openLogs.put(mainLogFile, log);
        if (reader == null) {
            reader = log.createReader();
            reader.setLazyDeserialization(deserializationPolicy == DeserializationPolicy.Lazy);
            interactiveTool.info("reader opened with " + deserializationPolicy + " deserialization.");
        }

        if (queryEngine != null) {
            RogLogQueryRepository repo = log.asRepository();
            repo.open();
            queryEngine.addRepository(repo, getLogNameAlias(log));
        }
    }

    final private RogLog createLog(File log, boolean repair, String openMode, Properties openProps) throws IOException, OdsException {
        File mainLogFile = mainLogFile(log);
        String name = mainLogFile.getName().substring(0, mainLogFile.getName().length() - 4);
        final Properties props = new Properties();
        props.setProperty("detachedPersist", "false");
        props.setProperty("storeRoot", mainLogFile.getParent());
        props.setProperty("autoRepair", String.valueOf(repair));
        props.setProperty("logMode", openMode);
        if (openProps != null) {
            props.putAll(openProps);
        }
        return RogLog.create(name, props);
    }

    final private void listOpen() throws IOException {
        listOpen(null, true);
    }

    final private void listOpen(String filter, boolean verbose) throws IOException {
        if (openLogs.isEmpty()) {
            interactiveTool.info("No logs are currently open. Use 'open' to open a log.");
            return;
        }

        if (filter == null) {
            interactiveTool.info("There are " + openLogs.size() + " open logs:");
            for (RogLog log : openLogs.values()) {
                listLog(log, verbose);
            }
        }
        else {
            interactiveTool.info("There are " + openLogs.size() + " open logs, matching against '" + filter + "':");
            for (RogLog log : openLogs.values()) {
                final String line = getLogNameAlias(log);
                if (filter != null && line.indexOf(filter) == -1) {
                    continue;
                }
                listLog(log, verbose);
            }
        }
    }

    final private void listLog(final RogLog log, final boolean verbose) throws IOException {
        interactiveTool.info(getLogNameAlias(log) + (verbose ? " [" + log.getLogFile().getCanonicalPath() + "]" : ""));
        if (verbose) {
            interactiveTool.info("  Log Version    : " + log.getMetadata().getVersion());
            interactiveTool.info("  Used Size      : " + UtlFile.readableFileSize(log.getSize()));
            interactiveTool.info("  File Size      : " + UtlFile.readableFileSize(log.getLogFile().length()));
            interactiveTool.info("  Free Disk      : " + UtlFile.readableFileSize(log.getLogFile().getFreeSpace()));
            interactiveTool.info("  Last Modified  : " + UtlTime.format(new Date(log.getLogFile().lastModified())));
            interactiveTool.info("  Validated FP   : " + log.getMetadata().getLastValidatedPosition());
            interactiveTool.info("  Log Number     : " + log.getMetadata().getLiveLogNumber());
            interactiveTool.info("  CDC Chkpt Ver  : " + log.getMetadata().getCdcCheckpointVersion());
            interactiveTool.info("  CDC Cursor     : " + log.getMetadata().getCdcCursor());
            interactiveTool.info("  CDC Log Number : " + log.getMetadata().getCdcLogNumber());
            if (queryEngine != null) {
                for (RogLogQueryRepository repo : queryEngine.getRepositories()) {
                    if (repo.getLog() == log) {
                        interactiveTool.info("  Num Indexes    :" + ((QueryIndexableRepository<?, ?>)repo).getIndexes().size());
                        interactiveTool.info("  Index Complete :" + ((QueryIndexableRepository<?, ?>)repo).isLiveIndexingUpToDate());
                    }
                }
            }
        }
    }

    final private boolean closeLogOrDirectory(File file) throws Exception {
        if (file.exists() && file.isDirectory()) {
            boolean closed = false;
            for (File log : file.listFiles(logMetadataFilter)) {
                closed |= closeLog(new File(file, log.getName().substring(0, log.getName().length() - 9)));
            }
            return closed;
        }
        else {
            return closeLog(file);
        }
    }

    private void closeQuery() throws Exception {
        closeQueryResult();
        if (mode == Mode.QUERY) {
            switchTo("browse");
        }
    }

    private void closeQueryResult() {
        if (queryResults != null) {
            queryResults.close();
            queryResults = null;
            resultHeaders.clear();
            resultRowFormats.clear();
            resultFooters.clear();
        }
    }

    final private boolean closeLog(File logfile) throws Exception {
        File mainLogFile = mainLogFile(logfile);
        RogLog log = openLogs.remove(mainLogFile);
        if (log == null) {
            interactiveTool.info("No log named '" + logfile.getName() + " to close");
            return false;
        }

        if (queryEngine != null) {
            interactiveTool.info("Removing log '" + getLogNameAlias(log) + "' from the query engine");
            if (queryEngine != null) {
                closeQuery();
                for (QueryRepository<Long, RogLogReader.Entry> repo : queryEngine.getRepositories()) {
                    if (repo instanceof RogLogQueryRepository && repo.getName() == log.getName()) {
                        RogLogQueryRepository logRepo = (RogLogQueryRepository)repo;
                        if (logRepo.getLog() == log) {
                            logRepo.close();
                        }
                    }
                }
                queryEngine.removeRepository(getLogNameAlias(log));
            }
        }

        log.close();

        if (reader != null && reader.log().getName() == log.getName()) {
            reader = null;
            if (!inQueryMode()) {
                updatePrompt();
            }
        }

        return true;
    }

    /**
     * Asserts that a log is open.
     * @throws Exception 
     */
    final private void assertOpen() throws Exception {
        if (reader == null || !reader.log().isOpen()) {
            throw new Exception("This command requires an open log. Please use the 'open' command");
        }
    }

    /**
     * Asserts that we are in query mode.
     * @throws Exception 
     */
    final private void assertQueryMode() throws Exception {
        if (mode != Mode.QUERY) {
            throw new Exception("This command is only valid in query mode. Use 'switch query' to enter query mode");
        }
    }

    /**
     * Asserts that we are in query mode.
     * @throws Exception 
     */
    final private void assertBrowseMode() throws Exception {
        if (mode != Mode.BROWSE) {
            throw new Exception("This command is only valid in browse mode. Use 'switch browse' to enter browse mode");
        }
    }

    /**
     * Asserts that we have a result set to work with
     * @throws Exception 
     */
    final private void assertResultSet() throws Exception {
        assertQueryMode();
        if (queryResults == null) {
            throw new Exception("In query mode this command is only valid with a result set. Run 'SELECT' first.");
        }
    }

    /**
     * Tests if we're in query mode.
     * @return True is in Query mode.
     */
    final private boolean inQueryMode() {
        return queryEngine != null && mode == Mode.QUERY;
    }

    private void writeEntry(RogLogReader.Entry entry, boolean detailed, boolean csv, boolean showLog, PrintStream out) throws IOException {
        BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(out));
        writeEntry(entry, detailed, csv, showLog, bw);
        bw.flush();
    }

    final private void writeEntry(final RogLogReader.Entry entry, boolean detailed, boolean csv, boolean showLog, final BufferedWriter bw) throws IOException {
        if (!detailed) {
            bw.write(entry.toString(csv));
            bw.newLine();
        }
        else {
            RogLogUtil.dumpLogEntryJson(entry, metadataDisplay != MetadataPolicy.Off, metadataDisplay != MetadataPolicy.Only, filterUnsetFields, jsonPrettyPrintStyle, bw);
            bw.newLine();
        }
    }

    private void writeResultSetRow(RogLogQueryResultSet resultSet, boolean detailed, boolean csv, boolean showLog, final PrintStream out) throws IOException {
        BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(out));
        writeResultSetRow(resultSet, detailed, csv, showLog, bw);
        bw.flush();
    }

    private void writeResultSetRow(RogLogQueryResultSet resultSet, boolean detailed, boolean csv, boolean showLog, final BufferedWriter bw) throws IOException {
        if (csv) {
            RogLogUtil.dumpResultSetRowCsv(resultSet, bw);
        }
        else if (detailed) {
            RogLogUtil.dumpResultSetRowJson(resultSet, metadataDisplay != MetadataPolicy.Off, metadataDisplay != MetadataPolicy.Only, filterUnsetFields, jsonPrettyPrintStyle, bw);
            bw.newLine();
        }
        else {
            RogLogUtil.dumpResultSetRowTabular(resultSet, bw);
        }
    }

    /**
     * @param out The print stream to which to write.
     * @throws IOException If there is an error writing to the print stream
     */
    final private void writeQueryResult(PrintStream out, boolean includeHeader, Format format) throws IOException {
        BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(out));
        writeQueryResult(bw, includeHeader, format);
        bw.flush();
    }

    /**
     * @param bw The buffered writer to which to write the results.
     * @throws IOException 
     */
    final private void writeQueryResult(final BufferedWriter bw, boolean includeHeader, Format format) throws IOException {
        String header = resultHeaders.get(format);
        if (header == null) {
            header = UtlQueryResultFormatter.getHeader(queryResults, format);
            resultHeaders.put(format, header);
            resultRowFormats.put(format, UtlQueryResultFormatter.getRowFormatString(queryResults, format));
            resultFooters.put(format, UtlQueryResultFormatter.getFooter(queryResults, format));
        }
        String rowFormat = resultRowFormats.get(format);

        if (includeHeader) {
            bw.write(header);
        }
        bw.write(UtlQueryResultFormatter.formatRow(queryResults, rowFormat, format));
    }

    /**
     * Open the log or logs at the given path. 
     * 
     * @param logPath The file or directory name of logs to open
     */
    final private Collection<RogLog> open(String logPath) {
        interactiveTool.executeCommand("open " + logPath);
        return openLogs.values();
    }

    /**
     * @return The collection of open {@link RogLog}s
     */
    final private Collection<RogLog> getOpenLogs() {
        return openLogs.values();
    }

    /**
     * Gets the tool's query engine.
     * 
     * @throws Exception If there is an error opening the query engine. 
     */
    final private RogLogQueryEngine getQueryEngine() throws Exception {
        switchTo("query");
        return queryEngine;
    }

    /**
     * Gets the name of the open log with the given id or null if 
     * no open log exists. 
     * 
     * @param logId The logId to locate
     * @return The log or null if no log is opened. 
     */
    final private String getLogFromUUID(String logId) {
        for (RogLog log : openLogs.values()) {
            if (log.getLogUUID().toString().equals(logId)) {
                return log.getName();
            }
        }
        return null;
    }

    /**
     * @return The list of discovered query types
     * @throws Exception Get the list of types.
     */
    final private Collection<Class<?>> getTypes() throws Exception {
        switchTo("query");
        return ((RogLogQueryEngineImpl)queryEngine).getTypes();
    }

    final private static void printUsage() {
        System.out.println("      'rumi tlt <args>' where args are:");
        System.err.println("          [{-e, --extractionDir the extraction folder to use for transaction log archives]");
        System.err.println("          [{-i, --initScripts <list-of scripts> to run at start} a comma separated list");
        System.err.println("                              of scripts to run before entering interactive mode or running");
        System.err.println("                              the main script.");
        System.err.println("          [{-s, --script <script name>} run the tool from a script]");
    }

    @Override
    final void help(final String[] args) {
        printUsage();
    }

    @Override
    final void execute(final String[] args) throws Exception {
        if (System.getProperty("nv.indent.json") == null) {
            System.setProperty("nv.indent.json", "true");
        }

        final CmdLineParser parser = new CmdLineParser();
        final CmdLineParser.Option extractionDirOption = parser.addStringOption('e', "extractionDir");
        final CmdLineParser.Option initScriptNameOption = parser.addStringOption('i', "initScripts");
        final CmdLineParser.Option scriptNameOption = parser.addStringOption('s', "script");
        File scriptFile = null;
        File extractionDir = null;
        parser.parse(args);
        final String scriptName = (String)parser.getOptionValue(scriptNameOption, null);
        scriptFile = scriptName == null ? null : new File(scriptName);
        if (scriptFile != null && !scriptFile.exists()) {
            throw new IllegalArgumentException("invalid script file '" + scriptFile + "'");
        }

        // if initScripts are specified on the command line they override existing
        // system/env properties:
        final String initScriptPaths = (String)parser.getOptionValue(initScriptNameOption, null);
        if (initScriptPaths != null) {
            System.getProperties().setProperty(PROP_TLT_INIT_SCRIPTS, initScriptPaths);
        }
        final String extractionDirName = (String)parser.getOptionValue(extractionDirOption, null);
        extractionDir = extractionDirName == null ? null : new File(extractionDirName);
        if (extractionDir != null) {
            if (!extractionDir.exists()) {
                if (!extractionDir.mkdirs()) {
                    throw new IllegalArgumentException("failed to create specified archive extraction dir '" + extractionDir + "'");
                }
            }
            else if (!extractionDir.isDirectory()) {
                throw new IllegalArgumentException("specified extraction dir '" + extractionDir + "' is not a directory.");
            }
        }
        if (scriptFile != null) {
            init(extractionDir).run(scriptFile);
        }
        else {
            init(extractionDir).run();
        }
    }
}
