/**
 * Copyright 2022 N5 Technologies, Inc
 *
 * This product includes software developed at N5 Technologies, Inc
 * (http://www.n5corp.com/) as well as software licenced to N5 Technologies,
 * Inc under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding
 * copyright ownership.
 *
 * N5 Technologies licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.neeve.tools;

import java.io.File;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import jargs.gnu.CmdLineParser;

import com.neeve.cloud.local.LocalProvisioner;

/**
 * The 'cloud local' command handler
 */
final class LocalCommand extends AbstractCommand {

    final private static class LaunchAdminCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local launch-admin' - launches Rumi admin");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().launchAdmin();
        }
    }

    final private static class StartAdminCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local start-admin' - starts Rumi admin");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().startAdmin();
        }
    }

    final private static class StopAdminCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local stop-admin' - stops Rumi admin");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().stopAdmin();
        }
    }

    final private static class TerminateAdminCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local terminate-admin' - terminates Rumi admin");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().terminateAdmin();
        }
    }

    final private static class LaunchMonitorCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local launch-monitor' - launches Rumi monitor");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().launchMonitor();
        }
    }

    final private static class StartMonitorCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local start-monitor' - starts Rumi monitor");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().startMonitor();
        }
    }

    final private static class StopMonitorCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local stop-monitor' - stops Rumi monitor");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().stopMonitor();
        }
    }

    final private static class TerminateMonitorCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local terminate-monitor' - terminates Rumi monitor");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().terminateMonitor();
        }
    }

    final private static class LaunchSolaceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local launch-solace' - launches Solace");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().launchSolaceBroker();
        }
    }

    final private static class StartSolaceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local start-solace' - starts Solace");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().startSolaceBroker();
        }
    }

    final private static class StopSolaceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local stop-solace' - stops Solace");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().stopSolaceBroker();
        }
    }

    final private static class TerminateSolaceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local terminate-solace' - terminates Solace");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().terminateSolaceBroker();
        }
    }

    final private static class LaunchKafkaCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local launch-kafka' - launches Kafka");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().launchKafkaBroker();
        }
    }

    final private static class StartKafkaCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local start-kafka' - starts Kafka");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().startKafkaBroker();
        }
    }

    final private static class StopKafkaCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local stop-kafka' - stops Kafka");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().stopKafkaBroker();
        }
    }

    final private static class TerminateKafkaCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local terminate-kafka' - terminates Kafka");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().terminateKafkaBroker();
        }
    }

    final private static class LaunchActiveMQCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local launch-amq' - launches ActiveMQ");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().launchActiveMQBroker();
        }
    }

    final private static class StartActiveMQCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local start-amq' - starts ActiveMQ");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().startActiveMQBroker();
        }
    }

    final private static class StopActiveMQCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local stop-amq' - stops ActiveMQ");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().stopActiveMQBroker();
        }
    }

    final private static class TerminateActiveMQCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local terminate-amq' - terminates ActiveMQ");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().terminateActiveMQBroker();
        }
    }

    final private static class LaunchPlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local launch-platform' - launches Rumi platform (admin and monitor)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().launchPlatform();
        }
    }

    final private static class StartPlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local start-platform' - starts Rumi platform (admin and monitor)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().startPlatform();
        }
    }

    final private static class StopPlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local stop-platform' - stops Rumi platform (admin and monitor)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().stopPlatform();
        }
    }

    final private static class TerminatePlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local terminate-platform' - terminates Rumi platform (admin and monitor)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().terminatePlatform();
        }
    }

    final private static class LaunchContainerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local launch-container <args>' where args are:");
            System.out.println("          [{-n, --name the name of the service container] (required)");
            System.out.println("          [{-h, --hostname the hostname for the service] (required)");
            System.out.println("          [{-p, --ports the comma separated set of service ports to expose] (default=null)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option hostnameOption = parser.addStringOption('h', "hostname");
            final CmdLineParser.Option portsOption = parser.addStringOption('p', "ports");
            parser.parse(args);
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String hostname = (String)parser.getOptionValue(hostnameOption, null);
            final String commaSeparatedPorts = (String)parser.getOptionValue(portsOption, null);
            if (name != null && hostname != null) {
                if (commaSeparatedPorts != null && !commaSeparatedPorts.trim().isEmpty()) {
                    final int[] ports = Arrays.stream(commaSeparatedPorts.split(","))
                                              .map(String::trim)
                                              .mapToInt(Integer::parseInt)
                                              .toArray();
                    LocalProvisioner.create().launchServiceContainer(name, hostname, ports);
                }
                else {
                    LocalProvisioner.create().launchServiceContainer(name, hostname);
                }
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartContainerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local start-container <args>' where args are:");
            System.out.println("          [{-n, --name the name of the service container] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            parser.parse(args);
            final String name = (String)parser.getOptionValue(nameOption, null);
            if (name != null) {
                LocalProvisioner.create().startServiceContainer(name);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopContainerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local stop-container <args>' where args are:");
            System.out.println("          [{-n, --name the name of the service container] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            parser.parse(args);
            final String name = (String)parser.getOptionValue(nameOption, null);
            if (name != null) {
                LocalProvisioner.create().stopServiceContainer(name);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateContainerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local terminate-container <args>' where args are:");
            System.out.println("          [{-n, --name the name of the service container] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            parser.parse(args);
            final String name = (String)parser.getOptionValue(nameOption, null);
            if (name != null) {
                LocalProvisioner.create().terminateServiceContainer(name);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class RunCommandCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local run-command <args>' where args are:");
            System.out.println("          [{-n, --name the name of the service container] (required)");
            System.out.println("          [{-c, --command the command to run] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option commandOption = parser.addStringOption('c', "command");
            parser.parse(args);
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String command = (String)parser.getOptionValue(commandOption, null);
            if (name != null && command != null) {
                LocalProvisioner.create().runCommandOnServiceContainer(name, command);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class ProvisionEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local provision <args>' where args are:");
            System.out.println("          [{-b, --message-broker the messaging broker to provision] (default=<no broker>)");
            System.out.println("                                 valid values: solace, kafka, activemq");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option messageBrokerOption = parser.addStringOption('b', "message-broker");
            parser.parse(args);
            final String messageBroker = (String)parser.getOptionValue(messageBrokerOption, null);

            if (messageBroker != null) {
                LocalProvisioner.create().provisionEnvironmentWithMessageBroker(messageBroker);
            }
            else {
                LocalProvisioner.create().provisionEnvironment();
            }
        }
    }

    final private static class DeploySystemCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local deploy <args>' where args are:");
            System.out.println("          [{-x, --xar the xar file to deploy] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option xarOption = parser.addStringOption('x', "xar");
            parser.parse(args);
            final String xarPath = (String)parser.getOptionValue(xarOption, null);
            if (xarPath != null) {
                LocalProvisioner.create().deploySystem(new File(xarPath));
            }
            else {
                printUsage();
            }
        }
    }

    final private static class ConfigureCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local configure <args>' where args are:");
            System.out.println("          [{-s, --section the configuration section] (required)");
            System.out.println("          [{-k, --key the property key] (required)");
            System.out.println("          [{-v, --value the property value (use 'null' to remove)] (required)");
            System.out.println("          [{-f, --file the config file path] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option sectionOption = parser.addStringOption('s', "section");
            final CmdLineParser.Option keyOption = parser.addStringOption('k', "key");
            final CmdLineParser.Option valueOption = parser.addStringOption('v', "value");
            final CmdLineParser.Option fileOption = parser.addStringOption('f', "file");
            parser.parse(args);
            final String section = (String)parser.getOptionValue(sectionOption, null);
            final String key = (String)parser.getOptionValue(keyOption, null);
            final String value = (String)parser.getOptionValue(valueOption, null);
            final String file = (String)parser.getOptionValue(fileOption, null);
            if (section != null && key != null && value != null && file != null) {
                LocalProvisioner.create().updateConfig(section, key, value, file);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local start' - starts all services in the local environment");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().startEnvironment();
        }
    }

    final private static class StopEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local stop' - stops all services in the local environment");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().stopEnvironment();
        }
    }

    final private static class DeprovisionEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud local deprovision' - deprovisions the local environment");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            LocalProvisioner.create().deprovisionEnvironment();
        }
    }

    final private static Set<String> commands = Stream.<String>of("launch-admin",
                                                                  "start-admin",
                                                                  "stop-admin",
                                                                  "terminate-admin",
                                                                  "launch-monitor",
                                                                  "start-monitor",
                                                                  "stop-monitor",
                                                                  "terminate-monitor",
                                                                  "launch-solace",
                                                                  "start-solace",
                                                                  "stop-solace",
                                                                  "terminate-solace",
                                                                  "launch-kafka",
                                                                  "start-kafka",
                                                                  "stop-kafka",
                                                                  "terminate-kafka",
                                                                  "launch-amq",
                                                                  "start-amq",
                                                                  "stop-amq",
                                                                  "terminate-amq",
                                                                  "launch-platform",
                                                                  "start-platform",
                                                                  "stop-platform",
                                                                  "terminate-platform",
                                                                  "launch-container",
                                                                  "start-container",
                                                                  "stop-container",
                                                                  "terminate-container",
                                                                  "run-command",
                                                                  "provision",
                                                                  "deploy",
                                                                  "configure",
                                                                  "start",
                                                                  "stop",
                                                                  "deprovision").collect(Collectors.toCollection(HashSet::new));

    final private void printTopLevelUsage() {
        System.out.println("");
        System.out.println("      Rumi 'cloud local' commands:");
        System.out.println("");
        System.out.println("         launch-admin           Provision and start Rumi admin");
        System.out.println("         start-admin            Start Rumi admin");
        System.out.println("         stop-admin             Stop Rumi admin");
        System.out.println("         terminate-admin        Terminate Rumi admin");
        System.out.println("         launch-monitor         Provision and start Rumi monitor");
        System.out.println("         start-monitor          Start Rumi monitor");
        System.out.println("         stop-monitor           Stop Rumi monitor");
        System.out.println("         terminate-monitor      Terminate Rumi monitor");
        System.out.println("         launch-solace          Provision and start Solace");
        System.out.println("         start-solace           Start Solace");
        System.out.println("         stop-solace            Stop Solace");
        System.out.println("         terminate-solace       Terminate Solace");
        System.out.println("         launch-kafka           Provision and start Kafka");
        System.out.println("         start-kafka            Start Kafka");
        System.out.println("         stop-kafka             Stop Kafka");
        System.out.println("         terminate-kafka        Terminate Kafka");
        System.out.println("         launch-amq             Provision and start ActiveMQ");
        System.out.println("         start-amq              Start ActiveMQ");
        System.out.println("         stop-amq               Stop ActiveMQ");
        System.out.println("         terminate-amq          Terminate ActiveMQ");
        System.out.println("         launch-platform        Provision and start Rumi platform (admin, monitor)");
        System.out.println("         start-platform         Start Rumi platform (admin, monitor)");
        System.out.println("         stop-platform          Stop Rumi platform (admin, monitor)");
        System.out.println("         terminate-platform     Terminate Rumi platform (admin, monitor)");
        System.out.println("         launch-container       Provision and start a Rumi service container");
        System.out.println("         start-container        Start a Rumi service container");
        System.out.println("         stop-container         Stop a Rumi service container");
        System.out.println("         terminate-container    Terminate a Rumi service container");
        System.out.println("         run-command            Run a command on a Rumi service container");
        System.out.println("         provision              Provision a local Rumi environment");
        System.out.println("         deploy                 Deploy a Rumi system to the local environment");
        System.out.println("         configure              Update a configuration property in the local environment");
        System.out.println("         start                  Start all services in the local environment");
        System.out.println("         stop                   Stop all services in the local environment");
        System.out.println("         deprovision            Deprovision the local Rumi environment");
        System.out.println("");
        System.out.println("      Run 'rumi help cloud local <command>' for help on a specific command");
        System.out.println("");
    }

    final private static AbstractCommand commandHandler(final String command) {
        if (command.equals("launch-admin")) {
            return new LaunchAdminCommand();
        }
        else if (command.equals("start-admin")) {
            return new StartAdminCommand();
        }
        else if (command.equals("stop-admin")) {
            return new StopAdminCommand();
        }
        else if (command.equals("terminate-admin")) {
            return new TerminateAdminCommand();
        }
        else if (command.equals("launch-monitor")) {
            return new LaunchMonitorCommand();
        }
        else if (command.equals("start-monitor")) {
            return new StartMonitorCommand();
        }
        else if (command.equals("stop-monitor")) {
            return new StopMonitorCommand();
        }
        else if (command.equals("terminate-monitor")) {
            return new TerminateMonitorCommand();
        }
        else if (command.equals("launch-solace")) {
            return new LaunchSolaceCommand();
        }
        else if (command.equals("start-solace")) {
            return new StartSolaceCommand();
        }
        else if (command.equals("stop-solace")) {
            return new StopSolaceCommand();
        }
        else if (command.equals("terminate-solace")) {
            return new TerminateSolaceCommand();
        }
        else if (command.equals("launch-kafka")) {
            return new LaunchKafkaCommand();
        }
        else if (command.equals("start-kafka")) {
            return new StartKafkaCommand();
        }
        else if (command.equals("stop-kafka")) {
            return new StopKafkaCommand();
        }
        else if (command.equals("terminate-kafka")) {
            return new TerminateKafkaCommand();
        }
        else if (command.equals("launch-amq")) {
            return new LaunchActiveMQCommand();
        }
        else if (command.equals("start-amq")) {
            return new StartActiveMQCommand();
        }
        else if (command.equals("stop-amq")) {
            return new StopActiveMQCommand();
        }
        else if (command.equals("terminate-amq")) {
            return new TerminateActiveMQCommand();
        }
        else if (command.equals("launch-platform")) {
            return new LaunchPlatformCommand();
        }
        else if (command.equals("start-platform")) {
            return new StartPlatformCommand();
        }
        else if (command.equals("stop-platform")) {
            return new StopPlatformCommand();
        }
        else if (command.equals("terminate-platform")) {
            return new TerminatePlatformCommand();
        }
        else if (command.equals("launch-container")) {
            return new LaunchContainerCommand();
        }
        else if (command.equals("start-container")) {
            return new StartContainerCommand();
        }
        else if (command.equals("stop-container")) {
            return new StopContainerCommand();
        }
        else if (command.equals("terminate-container")) {
            return new TerminateContainerCommand();
        }
        else if (command.equals("run-command")) {
            return new RunCommandCommand();
        }
        else if (command.equals("provision")) {
            return new ProvisionEnvironmentCommand();
        }
        else if (command.equals("deploy")) {
            return new DeploySystemCommand();
        }
        else if (command.equals("configure")) {
            return new ConfigureCommand();
        }
        else if (command.equals("start")) {
            return new StartEnvironmentCommand();
        }
        else if (command.equals("stop")) {
            return new StopEnvironmentCommand();
        }
        else if (command.equals("deprovision")) {
            return new DeprovisionEnvironmentCommand();
        }
        else {
            throw new IllegalArgumentException("unknown command '" + command + "'");
        }
    }

    @Override
    final void help(final String[] args) {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (commands.contains(command)) {
                commandHandler(command).help(null);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }

    @Override
    final void execute(final String[] args) throws Exception {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (commands.contains(command)) {
                commandHandler(command).execute(args.length > 1 ? Arrays.<String>copyOfRange(args, 1, args.length) : new String[0]);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }
}
