/**
 * Copyright 2022 N5 Technologies, Inc
 *
 * This product includes software developed at N5 Technologies, Inc
 * (http://www.n5corp.com/) as well as software licenced to N5 Technologies,
 * Inc under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding
 * copyright ownership.
 *
 * N5 Technologies licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.neeve.tools;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.lang.reflect.Method;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Properties;

import com.neeve.ods.IStoreObject;
import com.neeve.ods.IStoreObjectFactory;
import com.neeve.ods.StoreObjectFactoryRegistry;
import com.neeve.rog.IRogMessage;
import com.neeve.rog.log.RogLog;
import com.neeve.rog.log.RogLogFactory;
import com.neeve.rog.log.RogLogQuery;
import com.neeve.rog.log.RogLogQueryEngine;
import com.neeve.rog.log.RogLogReader;
import com.neeve.rog.log.RogLogQueryRepository;
import com.neeve.rog.log.RogLogQueryResultSet;
import com.neeve.server.mon.SrvMonHeartbeatFactory;
import com.neeve.server.mon.SrvMonHeartbeatMessage;
import com.neeve.server.mon.util.SrvMonHeartbeatTracer;
import com.neeve.util.UtlDataTypes;

import jargs.gnu.CmdLineParser;

/**
 * The "hbdump" command handler
 */
final class HeartbeatsDumpCommand extends AbstractCommand {
    final private static void printUsage() {
        System.out.println("      'rumi hbdump <args>' where args are:");
        System.err.println("          [{-l, --heartbeat-log <heartbeat log file>} the transaction log containing the heartbeat messages");
        System.err.println("          [{-o, --output-file <output text file>} the output file containing the textualized heartbeats]");
        System.err.println("          [{-a, --include-admin} whether to output heartbeats for Rumi internal administrative services as well]");
        System.err.println("          [{-s, --start-datetime <start date>} the start date-time of the heartbeats to process e.g. \"yyyy-MM-dd hh:mm:ss[.f...] [Z]\" or \"ten minutes ago\".]");
        System.err.println("          [{-e, --end-datetime <end date>} the end date-time of the heartbeats to process e.g. \"yyyy-MM-dd hh:mm:ss[.f...] [Z]\" or \"five minutes ago\".]");
        System.err.println("          [{-b, --no-sys-stats <no sys stats>} whether to exclude sys stats.]");
        System.err.println("          [{-c, --no-service-stats <no service stats>} whether to exclude service stats.]");
        System.err.println("          [{-d, --no-thread-stats <no thread stats>} whether to exclude thread stats.]");
        System.err.println("          [{-f, --no-user-stats <no user stats>} whether to exclude user stats.]");
        System.err.println("          [{-g, --no-pool-stats <no pool stats>} whether to exclude pool stats.]");
        System.err.println("          [{-i, --no-admin-client-stats <no admnin client stats>} whether to exclude admin client stats.]");
    }

    final private String buildWarmupTimeClauseForQuery(final String startDateTime) throws Exception {
        final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        final Calendar cal = Calendar.getInstance();
        cal.setTime(UtlDataTypes.parseAsDate(startDateTime));
        cal.add(Calendar.MINUTE, -1);
        return " WHERE timestamp BETWEEN '" + dateFormat.format(cal.getTime()) + "' AND '" + startDateTime + "'";
    }

    final private String buildTimeClauseForQuery(final String startDateTime, final String endDateTime) throws Exception {
        String queryString = "";
        if (startDateTime != null && endDateTime != null) {
            queryString = " WHERE timestamp BETWEEN '" + startDateTime + "' AND '" + endDateTime + "'";
        }
        else {
            if (startDateTime != null) {
                queryString += " WHERE timestamp > '" + startDateTime + "'";
            }
            if (endDateTime != null) {
                queryString += " WHERE timestamp < '" + endDateTime + "'";
            }
        }
        return queryString;
    }

    final private void warmup(final RogLogQueryEngine queryEngine, final String queryString, final SrvMonHeartbeatTracer heartbeatTracer) throws Exception {
        final RogLogQuery query = queryEngine.createQuery(queryString);
        final RogLogQueryResultSet results = queryEngine.execute(query);
        final StringBuilder sb = new StringBuilder();
        while (results.next()) {
            sb.setLength(0);
            final IStoreObject object = results.getLogEntry().getObject();
            if (object instanceof IRogMessage) {
                SrvMonHeartbeatMessage srvMessage = (SrvMonHeartbeatMessage)object;
                heartbeatTracer.printStats(srvMessage, sb);
            }
        }
    }

    final private void dumpHeartbeats(final RogLogQueryEngine queryEngine, 
                                      final String queryString, 
                                      final SrvMonHeartbeatTracer heartbeatTracer, 
                                      final File outFile) throws Exception {
        // run query
        final RogLogQuery query = queryEngine.createQuery(queryString);
        final RogLogQueryResultSet results = queryEngine.execute(query);

        // write results
        System.out.println("Writing " + results.getCount() + " heartbeats to " + outFile.getAbsoluteFile());
        final FileOutputStream fout = new FileOutputStream(outFile);
        try {
            final StringBuilder sb = new StringBuilder();
            while (results.next()) {
                sb.setLength(0);
                final IStoreObject object = results.getLogEntry().getObject();
                if (object instanceof IRogMessage) {
                    final SrvMonHeartbeatMessage heartbeat = (SrvMonHeartbeatMessage)object;
                    sb.append(String.format("\n[%s]<%s,%s> .....[STATS]\n", 
                                            new Date(heartbeat.getCollectionStartWallTime()), 
                                            heartbeat.getServerPid(), 
                                            heartbeat.getServerHostName()));
                    heartbeatTracer.printStats(heartbeat, sb);
                    fout.write(sb.toString().getBytes());
                }
            }
        }
        finally {
            fout.close();
        }
        System.out.println("Done.");
    }

    @Override
    final void help(final String[] args) {
        printUsage();
    }

    @Override
    final void execute(final String[] args) throws Exception {
        final CmdLineParser parser = new CmdLineParser();
        final CmdLineParser.Option logNameOption = parser.addStringOption('l', "heartbeat-log");
        final CmdLineParser.Option outputFileOption = parser.addStringOption('o', "output-file");
        final CmdLineParser.Option includeAdminOption = parser.addBooleanOption('a', "include-admin");
        final CmdLineParser.Option startOption = parser.addStringOption('s', "start-datetime");
        final CmdLineParser.Option endOption = parser.addStringOption('e', "end-datetime");
        final CmdLineParser.Option noSysStatsOption = parser.addBooleanOption('b', "no-sys-stats");
        final CmdLineParser.Option noServiceStatsOption = parser.addBooleanOption('c', "no-service-stats");
        final CmdLineParser.Option noThreadStatsOption = parser.addBooleanOption('c', "no-thread-stats");
        final CmdLineParser.Option noUsserStatsOption = parser.addBooleanOption('c', "no-user-stats");
        final CmdLineParser.Option noPoolStatsOption = parser.addBooleanOption('c', "no-pool-stats");
        final CmdLineParser.Option noAdminClientStatsOption = parser.addBooleanOption('c', "no-admin-client-stats");
        parser.parse(args);

        // log
        final String logName = (String)parser.getOptionValue(logNameOption, null);
        if (logName == null) {
            throw new IllegalArgumentException("heartbeat log needs to be specified");
        }
        File logFile = new File(logName);
        if (!logFile.exists()) {
            throw new IllegalArgumentException("heartbeat log '" + logFile + "' does not exist");
        }
        logFile = logFile.getAbsoluteFile();

        // output file
        final String outputFileName = (String)parser.getOptionValue(outputFileOption, "stats.txt");
        final File outputFile = new File(outputFileName);
        if (outputFile.exists()) {
            throw new IllegalArgumentException("output file '" + outputFile.getAbsoluteFile() + "' already exists");
        }

        // include admin
        final boolean includeAdmin = (Boolean)parser.getOptionValue(includeAdminOption, false);

        // start date
        final String startDateTime = (String)parser.getOptionValue(startOption, null);

        // end date
        final String endDateTime = (String)parser.getOptionValue(endOption, null);

        // stats selectors
        final boolean noSysStats = (Boolean)parser.getOptionValue(noSysStatsOption, false);
        final boolean noServiceStats = (Boolean)parser.getOptionValue(noServiceStatsOption, false);
        final boolean noThreadStats = (Boolean)parser.getOptionValue(noThreadStatsOption, false);
        final boolean noUserStats = (Boolean)parser.getOptionValue(noUsserStatsOption, false);
        final boolean noPoolStats = (Boolean)parser.getOptionValue(noPoolStatsOption, false);
        final boolean noAdminClientStats = (Boolean)parser.getOptionValue(noAdminClientStatsOption, false);

        // create heartbeat tracer
        final SrvMonHeartbeatTracer heartbeatTracer = new SrvMonHeartbeatTracer();
        heartbeatTracer.setFilterAdminApps(!includeAdmin);
        heartbeatTracer.setTraceSysStats(!noSysStats);
        heartbeatTracer.setTraceAppStats(!noServiceStats);
        heartbeatTracer.setTraceThreadStats(!noThreadStats);
        heartbeatTracer.setTraceUserStats(!noUserStats);
        heartbeatTracer.setTracePoolStats(!noPoolStats);
        heartbeatTracer.setTraceAdminClientStats(!noAdminClientStats);

        // register heartbeat factory
        StoreObjectFactoryRegistry.getInstance().registerObjectFactory(SrvMonHeartbeatFactory.class.getName());

        // open log 
        final String name = logFile.getName().substring(0, logFile.getName().length() - 4);
        final Properties props = new Properties();
        props.setProperty("storeRoot", logFile.getParent());
        props.setProperty("autoRepair", String.valueOf(false));
        props.setProperty("logMode", "r");
        final RogLog log = RogLogFactory.createLog(name, props);
        log.open();
        try {
            // create query engine
            final RogLogQueryEngine queryEngine = RogLogFactory.createQueryEngine();
            queryEngine.setAutoIndexing(false);
            try {
                final RogLogQueryRepository queryRepository = log.asRepository();
                queryRepository.open();
                try {
                    queryEngine.addRepository(queryRepository, name);

                    // build query
                    final String queryCore = "SELECT * from logs";
                    final String queryTimeClause = buildTimeClauseForQuery(startDateTime, endDateTime);

                    // if start date is specified, run a warmup query for a minute before
                    // ...this is done to calcuate any averages before actual query is done 
                    if (startDateTime != null) {
                        final String warmupQuery = buildWarmupTimeClauseForQuery(startDateTime);
                        if (warmupQuery != null) {
                            warmup(queryEngine, queryCore + warmupQuery, heartbeatTracer);
                        }
                    }

                    // now process file
                    dumpHeartbeats(queryEngine, queryCore + queryTimeClause, heartbeatTracer, outputFile);
                }
                finally {
                    queryRepository.close();
                }
            }
            finally {
                queryEngine.close(); 
            }
        }
        finally {
            log.close(); 
        }
    }
}
