#!/bin/bash

print_usage() {
    echo "usage: $0 [--help] [--port <listen_port>] [--host-name <target_host>] [--host-port <target_port>] [--server <server_name>]" >&2
    exit 1
}

# Default values
PORT=""
HOST_NAME=""
HOST_PORT=""
SERVER="_"

# Parse input parameters
while [[ "$1" == -* ]]; do
    case "$1" in
    --help|-h)
        print_usage
        ;;
    --port)
        shift
        PORT=$1
        shift
        ;;
    --host-name)
        shift
        HOST_NAME=$1
        shift
        ;;
    --host-port)
        shift
        HOST_PORT=$1
        shift
        ;;
    --server|-s)
        shift
        SERVER=$1
        shift
        ;;
    *)
        echo "error: unknown option: $1" >&2
        print_usage
        ;;
    esac
done

# Validate required inputs
if [[ -z "$HOST_NAME" || -z "$HOST_PORT" ]]; then
    print_usage
fi

# Default listen port to host port if not provided
if [[ -z "$PORT" ]]; then
    PORT="$HOST_PORT"
fi

# Wait for HOST_NAME to be reachable
TIMEOUT=300
INTERVAL=5
ELAPSED=0
echo "Waiting for $HOST_NAME to become reachable..."
while ! ping -c 1 "$HOST_NAME" &> /dev/null; do
    if [ "$ELAPSED" -ge "$TIMEOUT" ]; then
        echo "Timeout reached! $HOST_NAME is still not reachable."
        exit 1
    fi

    echo "Still waiting... Retrying in $INTERVAL seconds..."
    sleep "$INTERVAL"
    ELAPSED=$((ELAPSED + INTERVAL))
done
echo "$HOST_NAME is now reachable!"

# Proxy config
sudo sed -i "/include \/etc\/nginx\/conf\.d\/\*\.conf/a\\
\\
    server {\n\
        listen       $PORT;\n\
        listen       [::]:$PORT;\n\
        server_name  $SERVER;\n\
        root         /usr/share/nginx/html;\n\
\n\
        # Load configuration files for the default server block.\n\
        include /etc/nginx/default.d/*.conf;\n\
\n\
        location / {\n\
            proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;\n\
            proxy_set_header Host \$host;\n\
            proxy_http_version 1.1;\n\
            proxy_set_header Upgrade \$http_upgrade;\n\
            proxy_set_header Connection \"upgrade\";\n\
            client_max_body_size 100M;\n\
            proxy_pass http://$HOST_NAME:$HOST_PORT;\n\
        }\n\
\n\
        error_page 404 /404.html;\n\
        location = /404.html {\n\
        }\n\
\n\
        error_page 500 502 503 504 /50x.html;\n\
        location = /50x.html {\n\
        }\n\
    }" /etc/nginx/nginx.conf

# configure SE Linux to allow nginx to connect to the port it proxies
sudo semanage port -m -t http_port_t -p tcp $HOST_PORT
sudo setsebool -P httpd_can_network_connect 1
sudo restorecon -Rv /etc/nginx

# restart nginx
sudo systemctl restart nginx

