/**
 * Copyright 2022 N5 Technologies, Inc
 *
 * This product includes software developed at N5 Technologies, Inc
 * (http://www.n5corp.com/) as well as software licenced to N5 Technologies,
 * Inc under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding
 * copyright ownership.
 *
 * N5 Technologies licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.neeve.tools;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * The 'cloud' top level command handler
 */
final class CloudCommand extends AbstractCommand {
    final private static Set<String> topLevelCommands = Stream.<String>of("aws","civo","equinix","liquidweb","local").collect(Collectors.toCollection(HashSet::new));

    final private void printTopLevelUsage() {
        System.out.println("");
        System.out.println("      Rumi 'cloud' Commands:");
        System.out.println("");
        System.out.println("         aws        Commands that provision Rumi applications in AWS            (https://aws.amazon.com)");
        System.out.println("         civo       Commands that provision Rumi applications in Civo           (https://www.civo.com)");
        System.out.println("         equinix    Commands that provision Rumi applications in Equinix Metal  (https://deploy.equinix.com/)");
        System.out.println("         liquidweb  Commands that provision Rumi applications in LiquidWeb      (https://www.liquidweb.com)");
        System.out.println("         local      Commands that provision Rumi applications on local machine  (Docker-based)");
        System.out.println("");
        System.out.println("      Run 'rumi help cloud <command>' for help on a specific command");
        System.out.println("");
    }

    final private static AbstractCommand commandHandler(final String command) {
        if (command.equals("aws")) {
            return new AwsCommand();
        }
        else if (command.equals("civo")) {
            return new CivoCommand();
        }
        else if (command.equals("equinix")) {
            return new EquinixCommand();
        }
        else if (command.equals("liquidweb")) {
            return new LiquidwebCommand();
        }
        else if (command.equals("local")) {
            return new LocalCommand();
        }
        else {
            throw new IllegalArgumentException("unknown command '" + command + "'");
        }
    }

    @Override
    final void help(final String[] args) {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (topLevelCommands.contains(command)) {
                commandHandler(command).help(args.length > 1 ? Arrays.<String>copyOfRange(args, 1, args.length) : new String[0]);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }

    @Override
    final void execute(final String[] args) throws Exception {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (topLevelCommands.contains(command)) {
                commandHandler(command).execute(args.length > 1 ? Arrays.<String>copyOfRange(args, 1, args.length) : new String[0]);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }
}
