/**
 * Copyright 2022 N5 Technologies, Inc
 *
 * This product includes software developed at N5 Technologies, Inc
 * (http://www.n5corp.com/) as well as software licenced to N5 Technologies,
 * Inc under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding
 * copyright ownership.
 *
 * N5 Technologies licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.neeve.tools;

import java.io.File;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import jargs.gnu.CmdLineParser;

import com.neeve.cloud.aws.AwsProvisioner;
import com.neeve.cloud.aws.AwsProvisioner.MessageBroker;

/**
 * The 'cloud aws' command handler
 */
final class AwsCommand extends AbstractCommand {
    final private static class CreateNetworkCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws create-network <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the network (VPC)] (required)");
            System.out.println("          [{-a, --addr the network address in the form of x.y] (default=10.0)");
            System.out.println("          [{-o, --firewall-ports the comma separated set of ports to open in this network's firewall] (default=null)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option addrOption = parser.addStringOption('a', "addr");
            final CmdLineParser.Option firewallPortsOption = parser.addStringOption('o', "firewall-ports");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String addr = (String)parser.getOptionValue(addrOption, "10.0");
            final String commaSeparatedFirewallPorts = (String)parser.getOptionValue(firewallPortsOption, null);
            if (profile != null && name != null) {
                if (commaSeparatedFirewallPorts != null && !commaSeparatedFirewallPorts.trim().isEmpty()) {
                    AwsProvisioner.create(profile, region).createNetwork(name, 
                                                                         addr,
                                                                         Arrays.stream(commaSeparatedFirewallPorts.split(","))
                                                                               .map(String::trim)
                                                                               .map(Integer::parseInt)
                                                                               .collect(Collectors.toList()));
                }
                else {
                    AwsProvisioner.create(profile, region).createNetwork(name, addr);
                }
            }
            else {
                printUsage();
            }
        }
    }

    final private static class CreateZoneCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws create-zone <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network (VPC)] (required)");
            System.out.println("          [{-z, --zone the zone to create and associate with the network (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option zoneOption = parser.addStringOption('z', "zone");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String zone = (String)parser.getOptionValue(zoneOption, null);
            if (profile != null && network != null && zone != null) {
                AwsProvisioner.create(profile, region).createHostedZone(network, zone);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class ListZonesCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws create-zone <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network (VPC)] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            if (profile != null && network != null) {
                final List<String> zones = AwsProvisioner.create(profile, region).getHostedZones(network);
                System.out.println("");
                System.out.println("Provisioned Zones");
                System.out.println("-----------------");
                if (zones.size() > 0) {
                    final Iterator<String> iterator = zones.iterator();
                    while (iterator.hasNext()) {
                        System.out.println("  " + iterator.next());
                    }
                }
                else {
                    System.out.println("  <none>");
                }
                System.out.println("");
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchJumpServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws launch-jump <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).launchJumpServer(network, keyPair);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartJumpServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws start-jump <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).startJumpServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopJumpServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws stop-jump <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).stopJumpServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateJumpServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws terminate-jump <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).terminateJumpServer(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchAdminServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws launch-admin <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-a, --admin-proxy-server the admin proxy server] (default=null)");
            System.out.println("          [{-b, --admin-proxy-port the admin proxy port] (default=0)");
            System.out.println("          [{-c, --discovery-proxy-server the discovery proxy server] (default=null)");
            System.out.println("          [{-d, --discovery-proxy-port the discovery proxy port] (default=0)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option adminProxyServerOption = parser.addStringOption('a', "admin-proxy-server");
            final CmdLineParser.Option adminProxyPortOption = parser.addIntegerOption('b', "admin-proxy-port");
            final CmdLineParser.Option discoveryProxyServerOption = parser.addStringOption('c', "discovery-proxy-server");
            final CmdLineParser.Option discoveryProxyPortOption = parser.addIntegerOption('d', "discovery-proxy-port");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String adminProxyServer = (String)parser.getOptionValue(adminProxyServerOption, null);
            final int adminProxyPort = (Integer)parser.getOptionValue(adminProxyPortOption, 0);
            final String discoveryProxyServer = (String)parser.getOptionValue(discoveryProxyServerOption, null);
            final int discoveryProxyPort = (Integer)parser.getOptionValue(discoveryProxyPortOption, 0);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).launchAdminServer(network, 
                                                                         keyPair,
                                                                         adminProxyServer != null || adminProxyPort > 0 ? new AwsProvisioner.PlatformServiceProxyInfo(adminProxyServer, adminProxyPort) : null,
                                                                         discoveryProxyServer != null || discoveryProxyPort > 0 ? new AwsProvisioner.PlatformServiceProxyInfo(discoveryProxyServer, discoveryProxyPort) : null);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class RunAdminScriptCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws run-admin-script <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network containing the instance of which to run the command] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-s, --system the name of the system whose script to run] (required)");
            System.out.println("          [{-i, --script the script to run] (required)");
            System.out.println("          [{-a, --params the script params in key1=val1,key2=val2,...,keyN=valN formay] (default=null)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option systemOption = parser.addStringOption('s', "system");
            final CmdLineParser.Option scriptOption = parser.addStringOption('i', "script");
            final CmdLineParser.Option paramsOption = parser.addStringOption('a', "params");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String system = (String)parser.getOptionValue(systemOption, null);
            final String script = (String)parser.getOptionValue(scriptOption, null);
            final String params = (String)parser.getOptionValue(paramsOption, null);
            if (profile != null && network != null && keyPair != null && system != null && script != null) {
                AwsProvisioner.create(profile, region).runAdminScript(network, 
                                                                      keyPair, 
                                                                      system,
                                                                      script,
                                                                      params);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartAdminServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws start-admin <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).startAdminServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopAdminServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws stop-admin <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).stopAdminServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateAdminServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws terminate-admin <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).terminateAdminServer(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchMonitorServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws launch-monitor <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-a, --proxy-server the monitor proxy server] (default=null)");
            System.out.println("          [{-b, --proxy-port the monitor proxy port] (default=0)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option proxyServerOption = parser.addStringOption('a', "proxy-server");
            final CmdLineParser.Option proxyPortOption = parser.addIntegerOption('b', "proxy-port");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String proxyServer = (String)parser.getOptionValue(proxyServerOption, null);
            final int proxyPort = (Integer)parser.getOptionValue(proxyPortOption, 0);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).launchMonitorServer(network, 
                                                                           keyPair,
                                                                           proxyServer != null || proxyPort > 0 ? new AwsProvisioner.PlatformServiceProxyInfo(proxyServer, proxyPort) : null);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartMonitorServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws start-monitor <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).startMonitorServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopMonitorServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws stop-monitor <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).stopMonitorServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateMonitorServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws terminate-monitor <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-w, --wait wait for the instance to fully terminate] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).terminateMonitorServer(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchSolaceBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws launch-solace <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-a, --proxy-server the broker proxy server] (default=null)");
            System.out.println("          [{-b, --proxy-port the broker proxy port] (default=0)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option proxyServerOption = parser.addStringOption('a', "proxy-server");
            final CmdLineParser.Option proxyPortOption = parser.addIntegerOption('b', "proxy-port");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String proxyServer = (String)parser.getOptionValue(proxyServerOption, null);
            final int proxyPort = (Integer)parser.getOptionValue(proxyPortOption, 0);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).launchSolaceBroker(network, 
                                                                          keyPair,
                                                                          proxyServer != null || proxyPort > 0 ? new AwsProvisioner.PlatformServiceProxyInfo(proxyServer, proxyPort) : null);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartSolaceBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws start-solace <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).startSolaceBroker(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopSolaceBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws stop-solace <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).stopSolaceBroker(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateSolaceBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws terminate-solace <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).terminateSolaceBroker(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchKafkaBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws launch-kafka <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-a, --proxy-server the broker proxy server] (default=null)");
            System.out.println("          [{-b, --proxy-port the broker proxy port] (default=0)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option proxyServerOption = parser.addStringOption('a', "proxy-server");
            final CmdLineParser.Option proxyPortOption = parser.addIntegerOption('b', "proxy-port");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String proxyServer = (String)parser.getOptionValue(proxyServerOption, null);
            final int proxyPort = (Integer)parser.getOptionValue(proxyPortOption, 0);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).launchKafkaBroker(network, 
                                                                         keyPair,
                                                                         proxyServer != null || proxyPort > 0 ? new AwsProvisioner.PlatformServiceProxyInfo(proxyServer, proxyPort) : null);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartKafkaBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws start-kafka <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).startKafkaBroker(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopKafkaBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws stop-kafka <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).stopKafkaBroker(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateKafkaBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws terminate-kafka <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).terminateKafkaBroker(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchActiveMQBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws launch-amq <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-a, --proxy-server the broker proxy server] (default=null)");
            System.out.println("          [{-b, --proxy-port the broker proxy port] (default=0)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option proxyServerOption = parser.addStringOption('a', "proxy-server");
            final CmdLineParser.Option proxyPortOption = parser.addIntegerOption('b', "proxy-port");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String proxyServer = (String)parser.getOptionValue(proxyServerOption, null);
            final int proxyPort = (Integer)parser.getOptionValue(proxyPortOption, 0);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).launchActiveMQBroker(network, 
                                                                            keyPair,
                                                                            proxyServer != null || proxyPort > 0 ? new AwsProvisioner.PlatformServiceProxyInfo(proxyServer, proxyPort) : null);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartActiveMQBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws start-amq <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).startActiveMQBroker(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopActiveMQBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws stop-amq <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).stopActiveMQBroker(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateActiveMQBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws terminate-amq <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).terminateActiveMQBroker(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchPlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws launch-platform <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-a, --admin-proxy-server the admin proxy server] (default=null)");
            System.out.println("          [{-b, --admin-proxy-port the admin proxy port] (default=0)");
            System.out.println("          [{-c, --discovery-proxy-server the discovery proxy server] (default=null)");
            System.out.println("          [{-d, --discovery-proxy-port the discovery proxy port] (default=0)");
            System.out.println("          [{-e, --monitor-proxy-server the monitor proxy server] (default=null)");
            System.out.println("          [{-f, --monitor-proxy-port the monitor proxy port] (default=0)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option adminProxyServerOption = parser.addStringOption('a', "admin-proxy-server");
            final CmdLineParser.Option adminProxyPortOption = parser.addIntegerOption('b', "admin-proxy-port");
            final CmdLineParser.Option discoveryProxyServerOption = parser.addStringOption('c', "discovery-proxy-server");
            final CmdLineParser.Option discoveryProxyPortOption = parser.addIntegerOption('d', "discovery-proxy-port");
            final CmdLineParser.Option monitorProxyServerOption = parser.addStringOption('e', "monitor-proxy-server");
            final CmdLineParser.Option monitorProxyPortOption = parser.addIntegerOption('f', "monitor-proxy-port");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String adminProxyServer = (String)parser.getOptionValue(adminProxyServerOption, null);
            final int adminProxyPort = (Integer)parser.getOptionValue(adminProxyPortOption, 0);
            final String discoveryProxyServer = (String)parser.getOptionValue(discoveryProxyServerOption, null);
            final int discoveryProxyPort = (Integer)parser.getOptionValue(discoveryProxyPortOption, 0);
            final String monitorProxyServer = (String)parser.getOptionValue(monitorProxyServerOption, null);
            final int monitorProxyPort = (Integer)parser.getOptionValue(monitorProxyPortOption, 0);
            if (profile != null && network != null && keyPair != null) {
                final Map<AwsProvisioner.ProxiedService, AwsProvisioner.PlatformServiceProxyInfo> proxyInfo = new HashMap<AwsProvisioner.ProxiedService, AwsProvisioner.PlatformServiceProxyInfo>();
                if (adminProxyServer != null || adminProxyPort > 0) {
                    proxyInfo.put(AwsProvisioner.ProxiedService.admin, new AwsProvisioner.PlatformServiceProxyInfo(adminProxyServer, adminProxyPort));
                }
                if (discoveryProxyServer != null || discoveryProxyPort > 0) {
                    proxyInfo.put(AwsProvisioner.ProxiedService.discovery, new AwsProvisioner.PlatformServiceProxyInfo(discoveryProxyServer, discoveryProxyPort));
                }
                if (monitorProxyServer != null || monitorProxyPort > 0) {
                    proxyInfo.put(AwsProvisioner.ProxiedService.monitor, new AwsProvisioner.PlatformServiceProxyInfo(monitorProxyServer, monitorProxyPort));
                }
                AwsProvisioner.create(profile, region).launchPlatform(network, keyPair, proxyInfo.size() > 0 ? proxyInfo : null);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartPlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws start-platform <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).startPlatform(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopPlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws stop-platform <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null) {
                AwsProvisioner.create(profile, region).stopPlatform(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminatePlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws terminate-platform <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null && keyPair != null) {
                AwsProvisioner.create(profile, region).terminatePlatform(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchServiceInstanceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws launch-instance <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network in which to launch the instance] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-t  --type the type of the service instance to launch] (required)");
            System.out.println("          [{-i, --publicIP whether to assign a public IP to the instance] (default=false)");
            System.out.println("          [{-z, --zone the hosted zone to add this instance's DNS entry to] (default=rumi.local)");
            System.out.println("          [{-a, --name the name of the service instance to launch] (required)");
            System.out.println("          [{-o, --nameInZone the zone name of the service instance] (default=instance name in dotted case ('-' replaced with '.'))");
            System.out.println("          [{-l, --additionalNamesInZone the additional names to add the zone for this service instance] (default=null in which case no additional names are added to the zone for this instance)");
            System.out.println("          [{-v, --vsize the size, in GiB, of the additional (non-boot) volume on the instance] (default=32)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option typeOption = parser.addStringOption('t', "type");
            final CmdLineParser.Option publicIPOption = parser.addBooleanOption('i', "publicIP");
            final CmdLineParser.Option zoneOption = parser.addStringOption('z', "zone");
            final CmdLineParser.Option nameOption = parser.addStringOption('a', "name");
            final CmdLineParser.Option nameInZoneOption = parser.addStringOption('b', "nameInZone");
            final CmdLineParser.Option additionalNamesInZoneOption = parser.addStringOption('l', "additionalNamesInZone");
            final CmdLineParser.Option vsizeOption = parser.addIntegerOption('v', "vsize");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String type = (String)parser.getOptionValue(typeOption, null);
            final boolean publicIP = (Boolean)parser.getOptionValue(publicIPOption, false);
            final String zone = (String)parser.getOptionValue(zoneOption, "rumi.local");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String additionalNamesInZone = (String)parser.getOptionValue(additionalNamesInZoneOption, null);
            final int vsize = (Integer)parser.getOptionValue(vsizeOption, 32);
            if (profile != null && network != null && keyPair != null && name != null && type != null) {
                final String nameInZone = (String)parser.getOptionValue(nameInZoneOption, name.replace("-", "."));
                AwsProvisioner.create(profile, region).launchServiceInstance(network, 
                                                                             keyPair, 
                                                                             type, 
                                                                             publicIP, 
                                                                             zone, 
                                                                             name, 
                                                                             nameInZone, 
                                                                             additionalNamesInZone != null ? additionalNamesInZone.split(",") : null,
                                                                             vsize);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class AttachNewVolumeToServiceInstanceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws attach-volume <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-z, --zone the name of the hosted zone the service belongs to] (default=rumi.local)");
            System.out.println("          [{-i, --instanceName the name of the service instance to attach the volume to] (required)");
            System.out.println("          [{-o, --instanceNameInZone the zone name of the service instance] (default=null in which case the instance name is used as the name in the zone)");
            System.out.println("          [{-n  --volumeName the name to assign to the volume] (default=null in which case <noname> name is assigned to the volume)");
            System.out.println("          [{-t  --volumeType the volume type (GP3, SC1, ST1)] (default=GP3)");
            System.out.println("          [{-v, --volumeSize the size, in GiB, of the new volume] (default=1)");
            System.out.println("          [{-s, --snapshot the id of the snapshot to create the volume from] (default=null in which case a volume is created fresh)");
            System.out.println("          [{-m, --mountPoint the mount point] (default=null in which case the volume is not mounted)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option zoneOption = parser.addStringOption('z', "zone");
            final CmdLineParser.Option instanceNameOption = parser.addStringOption('i', "instanceName");
            final CmdLineParser.Option instanceNameInZoneOption = parser.addStringOption('j', "instanceNameInZone");
            final CmdLineParser.Option volumeNameOption = parser.addStringOption('n', "volumeName");
            final CmdLineParser.Option volumeTypeOption = parser.addStringOption('t', "volumeType");
            final CmdLineParser.Option volumeSizeOption = parser.addIntegerOption('i', "volumeSize");
            final CmdLineParser.Option snapshotOption = parser.addStringOption('s', "snapshot");
            final CmdLineParser.Option mountPointOption = parser.addStringOption('m', "mountPoint");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String zone = (String)parser.getOptionValue(zoneOption, "rumi.local");
            final String instanceName = (String)parser.getOptionValue(instanceNameOption, null);
            final String instanceNameInZone = (String)parser.getOptionValue(instanceNameInZoneOption, null);
            final String volumeName = (String)parser.getOptionValue(volumeNameOption, null);
            final String volumeType = (String)parser.getOptionValue(volumeTypeOption, "GP3");
            final int volumeSize = (Integer)parser.getOptionValue(volumeSizeOption, 1);
            final String snapshotId = (String)parser.getOptionValue(snapshotOption, null);
            final String mountPoint = (String)parser.getOptionValue(mountPointOption, null);
            if (profile != null && network != null && keyPair != null && instanceName != null) {
                AwsProvisioner.create(profile, region).attachNewVolumeToServiceInstance(network, 
                                                                                        keyPair, 
                                                                                        zone,
                                                                                        instanceName,
                                                                                        instanceNameInZone,
                                                                                        volumeName,
                                                                                        volumeType,
                                                                                        volumeSize,
                                                                                        snapshotId,
                                                                                        mountPoint);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class RunCommandOnServiceInstanceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws run-commmand <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network containing the instance of which to run the command] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-z, --zone the name of the hosted zone the instance belongs to] (default=rumi.local)");
            System.out.println("          [{-o, --instanceNameInZone the zone name of the instance] (required)");
            System.out.println("          [{-m, --command the command to run] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option zoneOption = parser.addStringOption('z', "zone");
            final CmdLineParser.Option instanceNameInZoneOption = parser.addStringOption('i', "instanceNameInZone");
            final CmdLineParser.Option commandOption = parser.addStringOption('c', "command");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String zone = (String)parser.getOptionValue(zoneOption, "rumi.local");
            final String instanceNameInZone = (String)parser.getOptionValue(instanceNameInZoneOption, null);
            final String command = (String)parser.getOptionValue(commandOption, null);
            if (profile != null && network != null && keyPair != null && instanceNameInZone != null && command != null) {
                AwsProvisioner.create(profile, region).runCommandOnServiceInstance(network, 
                                                                                   keyPair, 
                                                                                   zone,
                                                                                   instanceNameInZone,
                                                                                   command);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartServiceInstanceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws start-instance <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-a, --name the name of the service instance to start] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nameOption = parser.addStringOption('a', "name");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String name = (String)parser.getOptionValue(nameOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null && name != null) {
                AwsProvisioner.create(profile, region).startServiceInstance(network, name, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopServiceInstanceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws stop-instance <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-a, --name the name of the service instance to stop] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully stop] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nameOption = parser.addStringOption('a', "name");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String name = (String)parser.getOptionValue(nameOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null && name != null) {
                AwsProvisioner.create(profile, region).stopServiceInstance(network, name, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateServiceInstanceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws terminate-instance <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-i, --name the name of the service instance to terminate] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option nameOption = parser.addStringOption('i', "name");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String name = (String)parser.getOptionValue(nameOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && network != null && keyPair != null && name != null) {
                AwsProvisioner.create(profile, region).terminateServiceInstance(network, keyPair, name, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class ListInstancesCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws list-instances <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --network the name of the network to list instances in] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String network = (String)parser.getOptionValue(networkOption, null);
            if (profile != null) {
                final AwsProvisioner provisioner = AwsProvisioner.create(profile, region);
                List<String> instances = provisioner.getPlatformInstances(network);
                System.out.println("");
                System.out.println("Platform Instances");
                System.out.println("------------------");
                if (instances.size() > 0) {
                    final Iterator<String> iterator = instances.iterator();
                    while (iterator.hasNext()) {
                        System.out.println("  " + iterator.next());
                    }
                }
                else {
                    System.out.println("  <none>");
                }
                instances = provisioner.getMessageBrokerInstances(network);
                System.out.println("");
                System.out.println("Message Broker Instances");
                System.out.println("------------------------");
                if (instances.size() > 0) {
                    final Iterator<String> iterator = instances.iterator();
                    while (iterator.hasNext()) {
                        System.out.println("  " + iterator.next());
                    }
                }
                else {
                    System.out.println("  <none>");
                }
                instances = provisioner.getServiceInstances(network);
                System.out.println("");
                System.out.println("Service Instances");
                System.out.println("-----------------");
                if (instances.size() > 0) {
                    final Iterator<String> iterator = instances.iterator();
                    while (iterator.hasNext()) {
                        System.out.println("  " + iterator.next());
                    }
                }
                else {
                    System.out.println("  <none>");
                }
                System.out.println("");
            }
            else {
                printUsage();
            }
        }
    }

    final private static class ListNetworksCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws list-networks <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            if (profile != null) {
                final List<String> networks = AwsProvisioner.create(profile, region).getNetworks();
                System.out.println("");
                System.out.println("Provisioned Networks");
                System.out.println("--------------------");
                if (networks.size() > 0) {
                    final Iterator<String> iterator = networks.iterator();
                    while (iterator.hasNext()) {
                        System.out.println("  " + iterator.next());
                    }
                }
                else {
                    System.out.println("  <none>");
                }
                System.out.println("");
            }
            else {
                printUsage();
            }
        }
    }

    final private static class DeleteNetworkCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws delete-network <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the network (VPC)] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            if (profile != null && name != null && keyPair != null) {
                AwsProvisioner.create(profile, region).deleteNetwork(name, keyPair);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class ProvisionEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws provision <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the environment] (required)");
            System.out.println("          [{-a, --addr the address of the environment's network in the form of x.y] (default=10.0)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-b, --message-broker the messaging broker to provision too] (default=<no broker>)");
            System.out.println("          [{-o, --firewall-ports the comma separated set of ports to open in this network's firewall] (default=null)");
            System.out.println("          [{-a, --admin-proxy-server the admin proxy server] (default=null)");
            System.out.println("          [{-b, --admin-proxy-port the admin proxy port] (default=0)");
            System.out.println("          [{-c, --discovery-proxy-server the discovery proxy server] (default=null)");
            System.out.println("          [{-d, --discovery-proxy-port the discovery proxy port] (default=0)");
            System.out.println("          [{-e, --monitor-proxy-server the monitor proxy server] (default=null)");
            System.out.println("          [{-f, --monitor-proxy-port the monitor proxy port] (default=0)");
            System.out.println("          [{-g, --broker-proxy-server the broker proxy server] (default=null)");
            System.out.println("          [{-i, --broker-proxy-port the broker proxy port] (default=0)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option addrOption = parser.addStringOption('a', "addr");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option messageBrokerOption = parser.addStringOption('b', "message-broker");
            final CmdLineParser.Option firewallPortsOption = parser.addStringOption('o', "firewall-ports");
            final CmdLineParser.Option adminProxyServerOption = parser.addStringOption('a', "admin-proxy-server");
            final CmdLineParser.Option adminProxyPortOption = parser.addIntegerOption('b', "admin-proxy-port");
            final CmdLineParser.Option discoveryProxyServerOption = parser.addStringOption('c', "discovery-proxy-server");
            final CmdLineParser.Option discoveryProxyPortOption = parser.addIntegerOption('d', "discovery-proxy-port");
            final CmdLineParser.Option monitorProxyServerOption = parser.addStringOption('e', "monitor-proxy-server");
            final CmdLineParser.Option monitorProxyPortOption = parser.addIntegerOption('f', "monitor-proxy-port");
            final CmdLineParser.Option brokerProxyServerOption = parser.addStringOption('h', "broker-proxy-server");
            final CmdLineParser.Option brokerProxyPortOption = parser.addIntegerOption('i', "broker-proxy-port");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String addr = (String)parser.getOptionValue(addrOption, "10.0");
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String messageBrokerStr = (String)parser.getOptionValue(messageBrokerOption, null);
            final MessageBroker messageBroker = messageBrokerStr != null ? MessageBroker.valueOf(messageBrokerStr) : null;
            final String commaSeparatedFirewallPorts = (String)parser.getOptionValue(firewallPortsOption, null);
            final String adminProxyServer = (String)parser.getOptionValue(adminProxyServerOption, null);
            final int adminProxyPort = (Integer)parser.getOptionValue(adminProxyPortOption, 0);
            final String discoveryProxyServer = (String)parser.getOptionValue(discoveryProxyServerOption, null);
            final int discoveryProxyPort = (Integer)parser.getOptionValue(discoveryProxyPortOption, 0);
            final String monitorProxyServer = (String)parser.getOptionValue(monitorProxyServerOption, null);
            final int monitorProxyPort = (Integer)parser.getOptionValue(monitorProxyPortOption, 0);
            final String brokerProxyServer = (String)parser.getOptionValue(brokerProxyServerOption, null);
            final int brokerProxyPort = (Integer)parser.getOptionValue(brokerProxyPortOption, 0);
            if (profile != null && name != null && keyPair != null) {
                final List<Integer> firewallPorts;
                if ((commaSeparatedFirewallPorts != null && !commaSeparatedFirewallPorts.trim().isEmpty())) {
                    firewallPorts = Arrays.stream(commaSeparatedFirewallPorts.split(",")).map(String::trim).map(Integer::parseInt).collect(Collectors.toList());
                }
                else {
                    firewallPorts = null;
                }
                final Map<AwsProvisioner.ProxiedService, AwsProvisioner.PlatformServiceProxyInfo> platformProxyInfo = new HashMap<AwsProvisioner.ProxiedService, AwsProvisioner.PlatformServiceProxyInfo>();
                if (adminProxyServer != null || adminProxyPort > 0) {
                    platformProxyInfo.put(AwsProvisioner.ProxiedService.admin, new AwsProvisioner.PlatformServiceProxyInfo(adminProxyServer, adminProxyPort));
                }
                if (discoveryProxyServer != null || discoveryProxyPort > 0) {
                    platformProxyInfo.put(AwsProvisioner.ProxiedService.discovery, new AwsProvisioner.PlatformServiceProxyInfo(discoveryProxyServer, discoveryProxyPort));
                }
                if (monitorProxyServer != null || monitorProxyPort > 0) {
                    platformProxyInfo.put(AwsProvisioner.ProxiedService.monitor, new AwsProvisioner.PlatformServiceProxyInfo(monitorProxyServer, monitorProxyPort));
                }
                AwsProvisioner.create(profile, region).provisionEnvironmentWithMessageBroker(name, 
                                                                                             addr, 
                                                                                             keyPair, 
                                                                                             messageBroker, 
                                                                                             firewallPorts,
                                                                                             platformProxyInfo, 
                                                                                             messageBroker != null && (brokerProxyServer != null || brokerProxyPort > 0) ? new AwsProvisioner.PlatformServiceProxyInfo(brokerProxyServer, brokerProxyPort) : null);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class DeploySystemCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws deploy <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the environment to deploy the system to] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-x, --xar the xar file] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option xarOption = parser.addStringOption('x', "xar");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String xar = (String)parser.getOptionValue(xarOption, null);
            if (profile != null && keyPair != null && name != null && xar != null) {
                AwsProvisioner.create(profile, region).deploySystem(name, keyPair, new File(xar));
            }
            else {
                printUsage();
            }
        }
    }

    final private static class UpdateEnvironmentConfigCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws configure <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the environment whose config is to be updated] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("          [{-s, --section the configuration property section] (required)");
            System.out.println("          [{-e, --key the configuration property key] (required)");
            System.out.println("          [{-v, --value the configuration property value] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            final CmdLineParser.Option sectionOption = parser.addStringOption('s', "section");
            final CmdLineParser.Option keyOption = parser.addStringOption('e', "key");
            final CmdLineParser.Option valueOption = parser.addStringOption('v', "value");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String section = (String)parser.getOptionValue(sectionOption, null);
            final String key = (String)parser.getOptionValue(keyOption, null);
            final String value = (String)parser.getOptionValue(valueOption, null);
            if (profile != null && keyPair != null && name != null && section != null & key != null && value != null) {
                AwsProvisioner.create(profile, region).updateConfig(name, keyPair, section, key, value.equalsIgnoreCase("null") ? null : value);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws start <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the environment to start] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && name != null) {
                AwsProvisioner.create(profile, region).startEnvironment(name, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws stop <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the environment to stop] (required)");
            System.out.println("          [{-o, --no-wait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "no-wait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && name != null) {
                AwsProvisioner.create(profile, region).stopEnvironment(name, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class DeprovisionEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws deprovision <args>' where args are:");
            System.out.println("          [{-p, --profile the AWS profile to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the provisioned Rumi environment to deleted] (required)");
            System.out.println("          [{-y, --key-pair the network key pair] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "key-pair");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            if (profile != null && name != null && keyPair != null) {
                AwsProvisioner.create(profile, region).deprovisionEnvironment(name, keyPair);
            }
            else {
                printUsage();
            }
        }
    }

    final private static Set<String> commands = Stream.<String>of("create-network",
                                                                  "create-zone",
                                                                  "list-zones",
                                                                  "launch-jump",
                                                                  "start-jump",
                                                                  "stop-jump",
                                                                  "terminate-jump",
                                                                  "launch-admin",
                                                                  "run-admin-script",
                                                                  "start-admin",
                                                                  "stop-admin",
                                                                  "terminate-admin",
                                                                  "launch-monitor",
                                                                  "start-monitor",
                                                                  "stop-monitor",
                                                                  "terminate-monitor",
                                                                  "launch-solace",
                                                                  "start-solace",
                                                                  "stop-solace",
                                                                  "terminate-solace",
                                                                  "launch-kafka",
                                                                  "start-kafka",
                                                                  "stop-kafka",
                                                                  "terminate-kafka",
                                                                  "launch-amq",
                                                                  "start-amq",
                                                                  "stop-amq",
                                                                  "terminate-amq",
                                                                  "launch-platform",
                                                                  "start-platform",
                                                                  "stop-platform",
                                                                  "terminate-platform",
                                                                  "launch-instance",
                                                                  "attach-volume",
                                                                  "run-command",
                                                                  "start-instance",
                                                                  "stop-instance",
                                                                  "terminate-instance",
                                                                  "list-instances",
                                                                  "list-networks",
                                                                  "delete-network",
                                                                  "provision",
                                                                  "deploy",
                                                                  "configure",
                                                                  "start",
                                                                  "stop",
                                                                  "deprovision").collect(Collectors.toCollection(HashSet::new));

    final private void printTopLevelUsage() {
        System.out.println("");
        System.out.println("      Rumi 'cloud aws' commands:");
        System.out.println("");
        System.out.println("         create-network         Create a new Rumi network");
        System.out.println("         create-zone            Create a new hosted zone associated with a Rumi network");
        System.out.println("         list-zones             List the hosted zone associated with a Rumi network");
        System.out.println("         launch-jump            Provision and start the Rumi Jump Server");
        System.out.println("         start-jump             Start the Rumi Jump Server");
        System.out.println("         stop-jump              Stop the Rumi Jump Server");
        System.out.println("         terminate-jump         Terminate the Rumi Jump Server");
        System.out.println("         launch-admin           Provision and start the Rumi Admin Server");
        System.out.println("         run-admin-script       Run an Rumi admin script");
        System.out.println("         start-admin            Start the Rumi Admin Server");
        System.out.println("         stop-admin             Stop the Rumi Admin Server");
        System.out.println("         terminate-admin        Terminate the Rumi Admin Server");
        System.out.println("         launch-monitor         Provision and start the Rumi Monitor Server");
        System.out.println("         start-monitor          Start the Rumi Monitor Server");
        System.out.println("         stop-monitor           Stop the Rumi Monitor Server");
        System.out.println("         terminate-monitor      Terminate the Rumi Monitor Server");
        System.out.println("         launch-solace          Provision and start the Rumi Solace Broker");
        System.out.println("         start-solace           Start the Rumi Solace Broker");
        System.out.println("         stop-solace            Stop the Rumi Solace Broker");
        System.out.println("         terminate-solace       Terminate the Rumi Solace Broker");
        System.out.println("         launch-kafka           Provision and start the Rumi Kafka Broker");
        System.out.println("         start-kafka            Start the Rumi Kafka Broker");
        System.out.println("         stop-kafka             Stop the Rumi Kafka Broker");
        System.out.println("         terminate-kafka        Terminate the Rumi ActiveMQ Broker");
        System.out.println("         launch-amq             Provision and start the Rumi ActiveMQ Broker");
        System.out.println("         start-amq              Start the Rumi ActiveMQ Broker");
        System.out.println("         stop-amq               Stop the Rumi ActiveMQ Broker");
        System.out.println("         terminate-amq          Terminate the Rumi ActiveMQ Broker");
        System.out.println("         launch-platform        Provision and start  the Rumi platform components (Jump, Admin, Monitor, Messaging)");
        System.out.println("         start-platform         Start the Rumi platform components (Jump, Admin, Monitor, Messaging)");
        System.out.println("         stop-platform          Stop the Rumi platform components (Jump, Admin, Monitor, Messaging)");
        System.out.println("         terminate-platform     Terminate the Rumi platform components (Jump, Admin, Monitor, Messaging)");
        System.out.println("         launch-instance        Provision and start a Rumi service instance");
        System.out.println("         attach-volume          Provision, attach and mount a new volume on a Rumi service instance");
        System.out.println("         run-command            Run a custom command on a Rumi service instance");
        System.out.println("         start-instance         Start the Rumi service instance");
        System.out.println("         stop-instance          Stop the Rumi service instance");
        System.out.println("         terminate-instance     Terminate the Rumi service instance");
        System.out.println("         list-instances         List instances provisioned by Rumi");
        System.out.println("         list-networks          List networks provisioned by Rumi");
        System.out.println("         delete-network         Delete an existing Rumi network");
        System.out.println("         provision              Provision a Rumi environment in AWS");
        System.out.println("         deploy                 Deploy a Rumi system to a Rumi environment provisioned in the Civo cloud");
        System.out.println("         configure              Update a configuration property of a Rumi environment provisioned in the Civo cloud");
        System.out.println("         start                  Start all provisioned instances in an AWS Rumi environment");
        System.out.println("         stop                   Stop all provisioned instances in an AWS Rumi environment");
        System.out.println("         deprovision            Deprovision an an existing Rumi environment in AWS");
        System.out.println("");
        System.out.println("      Run 'rumi help cloud aws <command>' for help on a specific command");
        System.out.println("");
    }

    final private static AbstractCommand commandHandler(final String command) {
        if (command.equals("create-network")) {
            return new CreateNetworkCommand();
        }
        if (command.equals("create-zone")) {
            return new CreateZoneCommand();
        }
        if (command.equals("list-zones")) {
            return new ListZonesCommand();
        }
        else if (command.equals("launch-jump")) {
            return new LaunchJumpServerCommand();
        }
        else if (command.equals("start-jump")) {
            return new StartJumpServerCommand();
        }
        else if (command.equals("stop-jump")) {
            return new StopJumpServerCommand();
        }
        else if (command.equals("terminate-jump")) {
            return new TerminateJumpServerCommand();
        }
        else if (command.equals("launch-admin")) {
            return new LaunchAdminServerCommand();
        }
        else if (command.equals("run-admin-script")) {
            return new RunAdminScriptCommand();
        }
        else if (command.equals("start-admin")) {
            return new StartAdminServerCommand();
        }
        else if (command.equals("stop-admin")) {
            return new StopAdminServerCommand();
        }
        else if (command.equals("terminate-admin")) {
            return new TerminateAdminServerCommand();
        }
        else if (command.equals("launch-monitor")) {
            return new LaunchMonitorServerCommand();
        }
        else if (command.equals("start-monitor")) {
            return new StartMonitorServerCommand();
        }
        else if (command.equals("stop-monitor")) {
            return new StopMonitorServerCommand();
        }
        else if (command.equals("terminate-monitor")) {
            return new TerminateMonitorServerCommand();
        }
        else if (command.equals("launch-solace")) {
            return new LaunchSolaceBrokerCommand();
        }
        else if (command.equals("start-solace")) {
            return new StartSolaceBrokerCommand();
        }
        else if (command.equals("stop-solace")) {
            return new StopSolaceBrokerCommand();
        }
        else if (command.equals("terminate-solace")) {
            return new TerminateSolaceBrokerCommand();
        }
        else if (command.equals("launch-kafka")) {
            return new LaunchKafkaBrokerCommand();
        }
        else if (command.equals("start-kafka")) {
            return new StartKafkaBrokerCommand();
        }
        else if (command.equals("stop-kafka")) {
            return new StopKafkaBrokerCommand();
        }
        else if (command.equals("terminate-kafka")) {
            return new TerminateKafkaBrokerCommand();
        }
        else if (command.equals("launch-amq")) {
            return new LaunchActiveMQBrokerCommand();
        }
        else if (command.equals("start-amq")) {
            return new StartActiveMQBrokerCommand();
        }
        else if (command.equals("stop-amq")) {
            return new StopActiveMQBrokerCommand();
        }
        else if (command.equals("terminate-amq")) {
            return new TerminateActiveMQBrokerCommand();
        }
        else if (command.equals("launch-platform")) {
            return new LaunchPlatformCommand();
        }
        else if (command.equals("start-platform")) {
            return new StartPlatformCommand();
        }
        else if (command.equals("stop-platform")) {
            return new StopPlatformCommand();
        }
        else if (command.equals("terminate-platform")) {
            return new TerminatePlatformCommand();
        }
        else if (command.equals("launch-instance")) {
            return new LaunchServiceInstanceCommand();
        }
        else if (command.equals("attach-volume")) {
            return new AttachNewVolumeToServiceInstanceCommand();
        }
        else if (command.equals("run-command")) {
            return new RunCommandOnServiceInstanceCommand();
        }
        else if (command.equals("start-instance")) {
            return new StartServiceInstanceCommand();
        }
        else if (command.equals("stop-instance")) {
            return new StopServiceInstanceCommand();
        }
        else if (command.equals("terminate-instance")) {
            return new TerminateServiceInstanceCommand();
        }
        else if (command.equals("list-instances")) {
            return new ListInstancesCommand();
        }
        else if (command.equals("list-networks")) {
            return new ListNetworksCommand();
        }
        else if (command.equals("delete-network")) {
            return new DeleteNetworkCommand();
        }
        else if (command.equals("provision")) {
            return new ProvisionEnvironmentCommand();
        }
        else if (command.equals("deploy")) {
            return new DeploySystemCommand();
        }
        else if (command.equals("configure")) {
            return new UpdateEnvironmentConfigCommand();
        }
        else if (command.equals("start")) {
            return new StartEnvironmentCommand();
        }
        else if (command.equals("stop")) {
            return new StopEnvironmentCommand();
        }
        else if (command.equals("deprovision")) {
            return new DeprovisionEnvironmentCommand();
        }
        else {
            throw new IllegalArgumentException("unknown command '" + command + "'");
        }
    }

    @Override
    final void help(final String[] args) {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (commands.contains(command)) {
                commandHandler(command).help(null);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }

    @Override
    final void execute(final String[] args) throws Exception {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (commands.contains(command)) {
                commandHandler(command).execute(args.length > 1 ? Arrays.<String>copyOfRange(args, 1, args.length) : new String[0]);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }
}
