/*
 * Decompiled with CFR 0.152.
 */
package com.neeve.cloud.local;

import com.neeve.trace.Tracer;
import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.List;

public final class LocalProvisioner {
    private static final String NETWORK_NAME = "rumi";
    private static final String DISCOVERY_SHORT_NAME = "discovery";
    private static final String DISCOVERY_CONTAINER_NAME = "rumi-discovery";
    private static final String ADMIN_COMPOSE_URL = "https://downloads.n5corp.com/rumi/admin/latest/docker-compose.yml";
    private static final String[] ADMIN_VOLUMES = new String[]{"rumi-influxdb-data", "rumi-discovery-data", "rumi-agent-data", "rumi-admin-data"};
    private static final String ADMIN_SHORT_NAME = "admin";
    private static final String ADMIN_CONTAINER_NAME = "rumi-admin";
    private static final String ADMIN_SCRIPTS_DIR = "/home/rumi/scripts";
    private static final String AGENT_SHORT_NAME = "agent";
    private static final String AGENT_CONTAINER_NAME = "rumi-agent";
    private static final String MONITOR_SHORT_NAME = "monitor";
    private static final String MONITOR_CONTAINER_NAME = "rumi-monitor";
    private static final String MONITOR_VOLUME = "rumi-monitor-data";
    private static final String MONITOR_IMAGE = "neeve/nvx-rumi-monitor:latest";
    private static final String ACTIVEMQ_SHORT_NAME = "activemq";
    private static final String ACTIVEMQ_CONTAINER_NAME = "rumi-activemq";
    private static final String ACTIVEMQ_VOLUME = "rumi-activemq-data";
    private static final String ACTIVEMQ_IMAGE = "apache/activemq-classic:5.18.6";
    private static final String KAFKA_SHORT_NAME = "kafka";
    private static final String KAFKA_CONTAINER_NAME = "rumi-kafka";
    private static final String KAFKA_VOLUME = "rumi-kafka-data";
    private static final String KAFKA_IMAGE = "apache/kafka:latest";
    private static final String SOLACE_SHORT_NAME = "solace";
    private static final String SOLACE_CONTAINER_NAME = "rumi-solace";
    private static final String SOLACE_VOLUME = "rumi-solace-data";
    private static final String SOLACE_IMAGE = "solace/solace-pubsub-standard:latest";
    private static final String SERVICE_IMAGE = "neeve/nvx-rumi-worker:latest";
    private final Tracer tracer = Tracer.get((String)"nv.cloud");
    private File adminDockerComposeFile;

    private LocalProvisioner() {
    }

    public static LocalProvisioner create() {
        return new LocalProvisioner();
    }

    private void ensureDockerRunning() throws Exception {
        if (this.tracer.debug) {
            this.tracer.log("Checking if Docker is running...", Tracer.Level.DEBUG);
        }
        this.executeCommand(null, "docker", "info");
        if (this.tracer.debug) {
            this.tracer.log("Docker is running.", Tracer.Level.DEBUG);
        }
    }

    private void createDockerNetwork() throws Exception {
        this.tracer.log("...creating docker network 'rumi'.", Tracer.Level.VERBOSE);
        try {
            this.executeCommand(null, "docker", "network", "create", NETWORK_NAME);
        }
        catch (Exception e) {
            String output = this.executeCommandWithOutput(null, "docker", "network", "ls", "--filter", "name=rumi", "--format", "{{.Name}}");
            if (!output.contains(NETWORK_NAME)) {
                throw e;
            }
            this.tracer.log("...network already exists.", Tracer.Level.VERBOSE);
        }
    }

    private void createAdminDockerVolumes() throws Exception {
        for (String volume : ADMIN_VOLUMES) {
            this.tracer.log("...creating docker volume '" + volume + "'.", Tracer.Level.VERBOSE);
            try {
                this.executeCommand(null, "docker", "volume", "create", volume);
            }
            catch (Exception e) {
                String output = this.executeCommandWithOutput(null, "docker", "volume", "ls", "--filter", "name=" + volume, "--format", "{{.Name}}");
                if (!output.contains(volume)) {
                    throw e;
                }
                this.tracer.log("...volume already exists.", Tracer.Level.VERBOSE);
            }
        }
    }

    private void downloadAdminDockerCompose() throws Exception {
        this.tracer.log("...downloading admin docker-compose.yml.", Tracer.Level.VERBOSE);
        File tempDir = new File(System.getProperty("java.io.tmpdir"), ADMIN_CONTAINER_NAME);
        if (!tempDir.exists()) {
            tempDir.mkdirs();
        }
        this.adminDockerComposeFile = new File(tempDir, "docker-compose.yml");
        URL url = new URL(ADMIN_COMPOSE_URL);
        Files.copy(url.openStream(), this.adminDockerComposeFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    private void runAdminDockerCompose() throws Exception {
        this.tracer.log("...running admin docker-compose up.", Tracer.Level.VERBOSE);
        this.executeCommand(this.adminDockerComposeFile.getParentFile(), "docker-compose", "up", "-d");
    }

    private void copyScriptsToAdminContainers() throws Exception {
        this.tracer.log("...copying scripts to admin containers.", Tracer.Level.VERBOSE);
        this.executeCommand(null, "docker", "exec", "-u", NETWORK_NAME, AGENT_CONTAINER_NAME, "mkdir", "-p", ADMIN_SCRIPTS_DIR);
        String scriptsResourcePath = "/scripts/docker/admin";
        String runAdminScriptPath = this.getClass().getResource(scriptsResourcePath + "/run_admin_script.sh").getPath();
        this.executeCommand(null, "docker", "cp", runAdminScriptPath, "rumi-agent:/home/rumi/scripts/run_admin_script.sh");
        this.executeCommand(null, "docker", "exec", AGENT_CONTAINER_NAME, "chown", "rumi:rumi", "/home/rumi/scripts/run_admin_script.sh");
        this.executeCommand(null, "docker", "exec", AGENT_CONTAINER_NAME, "chmod", "+x", "/home/rumi/scripts/run_admin_script.sh");
        String updateConfigPath = this.getClass().getResource(scriptsResourcePath + "/update_config.sh").getPath();
        this.executeCommand(null, "docker", "cp", updateConfigPath, "rumi-agent:/home/rumi/scripts/update_config.sh");
        this.executeCommand(null, "docker", "exec", AGENT_CONTAINER_NAME, "chown", "rumi:rumi", "/home/rumi/scripts/update_config.sh");
        this.executeCommand(null, "docker", "exec", AGENT_CONTAINER_NAME, "chmod", "+x", "/home/rumi/scripts/update_config.sh");
    }

    private void createMonitorVolume() throws Exception {
        this.tracer.log("...creating docker volume 'rumi-monitor-data'.", Tracer.Level.VERBOSE);
        try {
            this.executeCommand(null, "docker", "volume", "create", MONITOR_VOLUME);
        }
        catch (Exception e) {
            String output = this.executeCommandWithOutput(null, "docker", "volume", "ls", "--filter", "name=rumi-monitor-data", "--format", "{{.Name}}");
            if (!output.contains(MONITOR_VOLUME)) {
                throw e;
            }
            this.tracer.log("...volume already exists.", Tracer.Level.VERBOSE);
        }
    }

    private void runMonitorContainer() throws Exception {
        this.tracer.log("...running monitor container.", Tracer.Level.VERBOSE);
        this.executeCommand(null, "docker", "run", "-d", "--name", MONITOR_CONTAINER_NAME, "--hostname", "monitor.rumi.local", "--network", NETWORK_NAME, "-p", "3000:3000", "-v", "rumi-monitor-data:/var/lib/grafana", "--restart", "unless-stopped", "--health-cmd", "curl -f http://localhost:3000/api/health", "--health-interval", "30s", "--health-timeout", "10s", "--health-retries", "3", "--health-start-period", "30s", MONITOR_IMAGE);
    }

    private void createActiveMQVolume() throws Exception {
        this.tracer.log("...creating docker volume 'rumi-activemq-data'.", Tracer.Level.VERBOSE);
        try {
            this.executeCommand(null, "docker", "volume", "create", ACTIVEMQ_VOLUME);
        }
        catch (Exception e) {
            String output = this.executeCommandWithOutput(null, "docker", "volume", "ls", "--filter", "name=rumi-activemq-data", "--format", "{{.Name}}");
            if (!output.contains(ACTIVEMQ_VOLUME)) {
                throw e;
            }
            this.tracer.log("...volume already exists.", Tracer.Level.VERBOSE);
        }
    }

    private void runActiveMQContainer() throws Exception {
        this.tracer.log("...running activemq broker container.", Tracer.Level.VERBOSE);
        this.executeCommand(null, "docker", "run", "-d", "--name", ACTIVEMQ_CONTAINER_NAME, "--hostname", "activemq.rumi.local", "--network", NETWORK_NAME, "-p", "61617:61616", "-p", "8162:8161", "-v", "rumi-activemq-data:/opt/apache-activemq/data", "--restart", "unless-stopped", "--health-cmd", "curl -f -u admin:admin http://localhost:8161", "--health-interval", "30s", "--health-timeout", "10s", "--health-retries", "5", "--health-start-period", "30s", ACTIVEMQ_IMAGE);
    }

    private void createKafkaVolume() throws Exception {
        this.tracer.log("...creating docker volume 'rumi-kafka-data'.", Tracer.Level.VERBOSE);
        try {
            this.executeCommand(null, "docker", "volume", "create", KAFKA_VOLUME);
        }
        catch (Exception e) {
            String output = this.executeCommandWithOutput(null, "docker", "volume", "ls", "--filter", "name=rumi-kafka-data", "--format", "{{.Name}}");
            if (!output.contains(KAFKA_VOLUME)) {
                throw e;
            }
            this.tracer.log("...volume already exists.", Tracer.Level.VERBOSE);
        }
    }

    private void runKafkaContainer() throws Exception {
        this.tracer.log("...running kafka broker container.", Tracer.Level.VERBOSE);
        this.executeCommand(null, "docker", "run", "-d", "--name", KAFKA_CONTAINER_NAME, "--hostname", "kafka.rumi.local", "--network", NETWORK_NAME, "-p", "9092:9092", "-v", "rumi-kafka-data:/var/lib/kafka/data", "--restart", "unless-stopped", "-e", "KAFKA_NODE_ID=1", "-e", "KAFKA_PROCESS_ROLES=broker,controller", "-e", "KAFKA_LISTENERS=PLAINTEXT://0.0.0.0:9092,CONTROLLER://0.0.0.0:9093", "-e", "KAFKA_ADVERTISED_LISTENERS=PLAINTEXT://kafka.rumi.local:9092", "-e", "KAFKA_CONTROLLER_LISTENER_NAMES=CONTROLLER", "-e", "KAFKA_LISTENER_SECURITY_PROTOCOL_MAP=CONTROLLER:PLAINTEXT,PLAINTEXT:PLAINTEXT", "-e", "KAFKA_CONTROLLER_QUORUM_VOTERS=1@kafka.rumi.local:9093", "-e", "KAFKA_OFFSETS_TOPIC_REPLICATION_FACTOR=1", "-e", "KAFKA_TRANSACTION_STATE_LOG_REPLICATION_FACTOR=1", "-e", "KAFKA_TRANSACTION_STATE_LOG_MIN_ISR=1", KAFKA_IMAGE);
    }

    private void createSolaceVolume() throws Exception {
        this.tracer.log("...creating docker volume 'rumi-solace-data'.", Tracer.Level.VERBOSE);
        try {
            this.executeCommand(null, "docker", "volume", "create", SOLACE_VOLUME);
        }
        catch (Exception e) {
            String output = this.executeCommandWithOutput(null, "docker", "volume", "ls", "--filter", "name=rumi-solace-data", "--format", "{{.Name}}");
            if (!output.contains(SOLACE_VOLUME)) {
                throw e;
            }
            this.tracer.log("...volume already exists.", Tracer.Level.VERBOSE);
        }
    }

    private void runSolaceContainer() throws Exception {
        this.tracer.log("...running solace broker container.", Tracer.Level.VERBOSE);
        this.executeCommand(null, "docker", "run", "-d", "--name", SOLACE_CONTAINER_NAME, "--hostname", "solace.rumi.local", "--network", NETWORK_NAME, "-p", "8080:8080", "-p", "55555:55555", "-p", "55003:55003", "-p", "8008:8008", "-p", "1883:1883", "-p", "5672:5672", "-p", "9000:9000", "-v", "rumi-solace-data:/var/lib/solace", "--shm-size", "2g", "--restart", "unless-stopped", "-e", "username_admin_globalaccesslevel=admin", "-e", "username_admin_password=admin", SOLACE_IMAGE);
    }

    private void createServiceVolume(String name) throws Exception {
        String volumeName = "rumi-" + name + "-data";
        this.tracer.log("...creating docker volume '" + volumeName + "'.", Tracer.Level.VERBOSE);
        try {
            this.executeCommand(null, "docker", "volume", "create", volumeName);
        }
        catch (Exception e) {
            String output = this.executeCommandWithOutput(null, "docker", "volume", "ls", "--filter", "name=" + volumeName, "--format", "{{.Name}}");
            if (!output.contains(volumeName)) {
                throw e;
            }
            this.tracer.log("...volume already exists.", Tracer.Level.VERBOSE);
        }
    }

    private void runServiceContainer(String name, String hostname, int[] servicePorts) throws Exception {
        String containerName = "rumi-" + name;
        String fullHostname = hostname + ".rumi.local";
        String volumeName = "rumi-" + name + "-data";
        this.tracer.log("...running service container '" + name + "'.", Tracer.Level.VERBOSE);
        ArrayList<String> command = new ArrayList<String>();
        command.add("docker");
        command.add("run");
        command.add("-d");
        command.add("--name");
        command.add(containerName);
        command.add("--hostname");
        command.add(fullHostname);
        command.add("--network");
        command.add(NETWORK_NAME);
        command.add("-v");
        command.add(volumeName + ":/home/rumi");
        if (servicePorts != null) {
            for (int port : servicePorts) {
                command.add("-p");
                command.add(port + ":" + port);
            }
        }
        command.add("--restart");
        command.add("unless-stopped");
        command.add(SERVICE_IMAGE);
        this.executeCommand(null, command.toArray(new String[0]));
    }

    private boolean isServiceContainer(String containerName) {
        return !containerName.equals(DISCOVERY_CONTAINER_NAME) && !containerName.equals(AGENT_CONTAINER_NAME) && !containerName.equals(ADMIN_CONTAINER_NAME) && !containerName.equals(MONITOR_CONTAINER_NAME) && !containerName.equals(ACTIVEMQ_CONTAINER_NAME) && !containerName.equals(KAFKA_CONTAINER_NAME) && !containerName.equals(SOLACE_CONTAINER_NAME);
    }

    private String getShortName(String containerName) {
        String prefix = "rumi-";
        if (containerName.startsWith("rumi-")) {
            return containerName.substring("rumi-".length());
        }
        return containerName;
    }

    private boolean containerExists(String containerName) throws Exception {
        try {
            String output = this.executeCommandWithOutput(null, "docker", "ps", "-a", "--filter", "name=^" + containerName + "$", "--format", "{{.Names}}");
            return output.trim().equals(containerName);
        }
        catch (Exception e) {
            return false;
        }
    }

    private boolean containerIsRunning(String containerName) throws Exception {
        try {
            String output = this.executeCommandWithOutput(null, "docker", "ps", "--filter", "name=^" + containerName + "$", "--format", "{{.Names}}");
            return output.trim().equals(containerName);
        }
        catch (Exception e) {
            return false;
        }
    }

    private void executeCommand(File workingDir, String ... command) throws Exception {
        String line;
        ProcessBuilder pb = new ProcessBuilder(command);
        if (workingDir != null) {
            pb.directory(workingDir);
        }
        pb.redirectErrorStream(true);
        Process process = pb.start();
        BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
        while ((line = reader.readLine()) != null) {
            this.tracer.log(line, Tracer.Level.INFO);
        }
        int exitCode = process.waitFor();
        if (exitCode != 0) {
            throw new RuntimeException("Command failed with exit code " + exitCode + ": " + String.join((CharSequence)" ", command));
        }
    }

    private String executeCommandWithOutput(File workingDir, String ... command) throws Exception {
        String line;
        ProcessBuilder pb = new ProcessBuilder(command);
        if (workingDir != null) {
            pb.directory(workingDir);
        }
        pb.redirectErrorStream(true);
        Process process = pb.start();
        StringBuilder output = new StringBuilder();
        BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
        while ((line = reader.readLine()) != null) {
            output.append(line).append("\n");
        }
        int exitCode = process.waitFor();
        if (exitCode != 0) {
            throw new RuntimeException("Command failed with exit code " + exitCode + ": " + String.join((CharSequence)" ", command));
        }
        return output.toString();
    }

    public final LocalProvisioner launchAdmin() {
        try {
            this.tracer.log("Launching rumi admin.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.createAdminDockerVolumes();
            this.downloadAdminDockerCompose();
            this.runAdminDockerCompose();
            this.copyScriptsToAdminContainers();
            this.tracer.log("rumi admin launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch rumi admin: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch admin", e);
        }
        return this;
    }

    public final LocalProvisioner startAdmin() {
        try {
            this.tracer.log("Starting rumi admin.", Tracer.Level.INFO);
            if (this.adminDockerComposeFile == null || !this.adminDockerComposeFile.exists()) {
                throw new IllegalStateException("Admin docker compose file not found. Please run launchAdmin first.");
            }
            this.executeCommand(this.adminDockerComposeFile.getParentFile(), "docker-compose", "start");
            this.tracer.log("rumi admin started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start rumi admin: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start admin", e);
        }
        return this;
    }

    public final LocalProvisioner runAdminScript(String system, String script, String params) {
        if (system == null) {
            throw new IllegalArgumentException("system name cannot be null");
        }
        if (script == null) {
            throw new IllegalArgumentException("script name cannot be null");
        }
        try {
            this.tracer.log("Executing admin script '" + script + "' on the '" + system + "' system with '" + params + "'.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "exec", AGENT_CONTAINER_NAME, "/home/rumi/scripts/run_admin_script.sh", "--system", system, "--script", script, "--param", params == null ? "dummy=value" : params);
            this.tracer.log("Successfully run admin script.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to run admin script: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to run admin script", e);
        }
        return this;
    }

    public final LocalProvisioner stopAdmin() {
        try {
            this.tracer.log("Stopping rumi admin.", Tracer.Level.INFO);
            if (this.adminDockerComposeFile == null || !this.adminDockerComposeFile.exists()) {
                throw new IllegalStateException("Admin docker compose file not found. Please run launchAdmin first.");
            }
            this.executeCommand(this.adminDockerComposeFile.getParentFile(), "docker-compose", "stop");
            this.tracer.log("rumi admin stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop rumi admin: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop admin", e);
        }
        return this;
    }

    public final LocalProvisioner terminateAdmin() {
        try {
            this.tracer.log("Terminating rumi admin.", Tracer.Level.INFO);
            if (this.adminDockerComposeFile != null && this.adminDockerComposeFile.exists()) {
                this.executeCommand(this.adminDockerComposeFile.getParentFile(), "docker-compose", "down");
                this.adminDockerComposeFile.delete();
                this.adminDockerComposeFile = null;
            }
            this.tracer.log("rumi admin terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate rumi admin: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate admin", e);
        }
        return this;
    }

    public final LocalProvisioner launchMonitor() {
        try {
            this.tracer.log("Launching the rumi monitor.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.createMonitorVolume();
            this.runMonitorContainer();
            this.tracer.log("rumi monitor launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch rumi monitor: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch monitor", e);
        }
        return this;
    }

    public final LocalProvisioner startMonitor() {
        try {
            this.tracer.log("Starting the rumi monitor.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "start", MONITOR_CONTAINER_NAME);
            this.tracer.log("rumi monitor started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start rumi monitor: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start monitor", e);
        }
        return this;
    }

    public final LocalProvisioner stopMonitor() {
        try {
            this.tracer.log("Stopping the rumi monitor.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "stop", MONITOR_CONTAINER_NAME);
            this.tracer.log("rumi monitor stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop rumi monitor: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop monitor", e);
        }
        return this;
    }

    public final LocalProvisioner terminateMonitor() {
        try {
            this.tracer.log("Terminating the rumi monitor.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "rm", "-f", MONITOR_CONTAINER_NAME);
            this.tracer.log("rumi monitor terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate rumi monitor: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate monitor", e);
        }
        return this;
    }

    public final LocalProvisioner launchSolaceBroker() {
        try {
            this.tracer.log("Launching solace.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.createSolaceVolume();
            this.runSolaceContainer();
            this.tracer.log("Solace launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch solace: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch solace", e);
        }
        return this;
    }

    public final LocalProvisioner startSolaceBroker() {
        try {
            this.tracer.log("Starting solace.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "start", SOLACE_CONTAINER_NAME);
            this.tracer.log("Solace started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start solace: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start solace", e);
        }
        return this;
    }

    public final LocalProvisioner stopSolaceBroker() {
        try {
            this.tracer.log("Stopping solace.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "stop", SOLACE_CONTAINER_NAME);
            this.tracer.log("Solace stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop solace: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop solace", e);
        }
        return this;
    }

    public final LocalProvisioner terminateSolaceBroker() {
        try {
            this.tracer.log("Terminating solace.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "rm", "-f", SOLACE_CONTAINER_NAME);
            this.tracer.log("Solace terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate solace: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate solace", e);
        }
        return this;
    }

    public final LocalProvisioner launchKafkaBroker() {
        try {
            this.tracer.log("Launching kafka.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.createKafkaVolume();
            this.runKafkaContainer();
            this.tracer.log("Kafka launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch kafka: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch kafka", e);
        }
        return this;
    }

    public final LocalProvisioner startKafkaBroker() {
        try {
            this.tracer.log("Starting kafka.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "start", KAFKA_CONTAINER_NAME);
            this.tracer.log("Kafka started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start kafka: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start kafka", e);
        }
        return this;
    }

    public final LocalProvisioner stopKafkaBroker() {
        try {
            this.tracer.log("Stopping kafka.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "stop", KAFKA_CONTAINER_NAME);
            this.tracer.log("Kafka stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop kafka: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop kafka", e);
        }
        return this;
    }

    public final LocalProvisioner terminateKafkaBroker() {
        try {
            this.tracer.log("Terminating kafka.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "rm", "-f", KAFKA_CONTAINER_NAME);
            this.tracer.log("Kafka terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate kafka: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate kafka", e);
        }
        return this;
    }

    public final LocalProvisioner launchActiveMQBroker() {
        try {
            this.tracer.log("Launching activemq.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.createActiveMQVolume();
            this.runActiveMQContainer();
            this.tracer.log("ActiveMQ launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch activemq: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch activemq", e);
        }
        return this;
    }

    public final LocalProvisioner startActiveMQBroker() {
        try {
            this.tracer.log("Starting activemq.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "start", ACTIVEMQ_CONTAINER_NAME);
            this.tracer.log("ActiveMQ started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start activemq: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start activemq", e);
        }
        return this;
    }

    public final LocalProvisioner stopActiveMQBroker() {
        try {
            this.tracer.log("Stopping activemq.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "stop", ACTIVEMQ_CONTAINER_NAME);
            this.tracer.log("ActiveMQ stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop activemq: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop activemq", e);
        }
        return this;
    }

    public final LocalProvisioner terminateActiveMQBroker() {
        try {
            this.tracer.log("Terminating activemq.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "rm", "-f", ACTIVEMQ_CONTAINER_NAME);
            this.tracer.log("ActiveMQ terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate activemq: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate activemq", e);
        }
        return this;
    }

    public final List<String> getAllMessageBrokers() {
        try {
            ArrayList<String> brokers = new ArrayList<String>();
            if (this.containerExists(ACTIVEMQ_CONTAINER_NAME)) {
                brokers.add(this.getShortName(ACTIVEMQ_CONTAINER_NAME));
            }
            if (this.containerExists(KAFKA_CONTAINER_NAME)) {
                brokers.add(this.getShortName(KAFKA_CONTAINER_NAME));
            }
            if (this.containerExists(SOLACE_CONTAINER_NAME)) {
                brokers.add(this.getShortName(SOLACE_CONTAINER_NAME));
            }
            return brokers;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get message brokers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get message brokers", e);
        }
    }

    public final List<String> getRunningMessageBrokers() {
        try {
            ArrayList<String> brokers = new ArrayList<String>();
            if (this.containerIsRunning(ACTIVEMQ_CONTAINER_NAME)) {
                brokers.add(this.getShortName(ACTIVEMQ_CONTAINER_NAME));
            }
            if (this.containerIsRunning(KAFKA_CONTAINER_NAME)) {
                brokers.add(this.getShortName(KAFKA_CONTAINER_NAME));
            }
            if (this.containerIsRunning(SOLACE_CONTAINER_NAME)) {
                brokers.add(this.getShortName(SOLACE_CONTAINER_NAME));
            }
            return brokers;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get running message brokers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get running message brokers", e);
        }
    }

    public final List<String> getStoppedMessageBrokers() {
        try {
            ArrayList<String> brokers = new ArrayList<String>();
            if (this.containerExists(ACTIVEMQ_CONTAINER_NAME) && !this.containerIsRunning(ACTIVEMQ_CONTAINER_NAME)) {
                brokers.add(this.getShortName(ACTIVEMQ_CONTAINER_NAME));
            }
            if (this.containerExists(KAFKA_CONTAINER_NAME) && !this.containerIsRunning(KAFKA_CONTAINER_NAME)) {
                brokers.add(this.getShortName(KAFKA_CONTAINER_NAME));
            }
            if (this.containerExists(SOLACE_CONTAINER_NAME) && !this.containerIsRunning(SOLACE_CONTAINER_NAME)) {
                brokers.add(this.getShortName(SOLACE_CONTAINER_NAME));
            }
            return brokers;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get stopped message brokers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get stopped message brokers", e);
        }
    }

    public final LocalProvisioner startAllMessageBrokers() {
        try {
            for (String broker : this.getStoppedMessageBrokers()) {
                if (ACTIVEMQ_SHORT_NAME.equals(broker)) {
                    this.startActiveMQBroker();
                    continue;
                }
                if (KAFKA_SHORT_NAME.equals(broker)) {
                    this.startKafkaBroker();
                    continue;
                }
                if (SOLACE_SHORT_NAME.equals(broker)) {
                    this.startSolaceBroker();
                    continue;
                }
                this.tracer.log("Encountered unknown message broker type '" + broker + "'. Ignoring...", Tracer.Level.WARNING);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to start message broker instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start message broker instances", e);
        }
        return this;
    }

    public final LocalProvisioner stopAllMessageBrokers() {
        try {
            for (String broker : this.getRunningMessageBrokers()) {
                if (ACTIVEMQ_SHORT_NAME.equals(broker)) {
                    this.stopActiveMQBroker();
                    continue;
                }
                if (KAFKA_SHORT_NAME.equals(broker)) {
                    this.stopKafkaBroker();
                    continue;
                }
                if (SOLACE_SHORT_NAME.equals(broker)) {
                    this.stopSolaceBroker();
                    continue;
                }
                this.tracer.log("Encountered unknown message broker type '" + broker + "'. Ignoring...", Tracer.Level.WARNING);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop message broker instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop message broker instances", e);
        }
        return this;
    }

    public final LocalProvisioner terminateAllMessageBrokers() {
        try {
            for (String broker : this.getAllMessageBrokers()) {
                if (ACTIVEMQ_SHORT_NAME.equals(broker)) {
                    this.terminateActiveMQBroker();
                    continue;
                }
                if (KAFKA_SHORT_NAME.equals(broker)) {
                    this.terminateKafkaBroker();
                    continue;
                }
                if (SOLACE_SHORT_NAME.equals(broker)) {
                    this.terminateSolaceBroker();
                    continue;
                }
                this.tracer.log("Encountered unknown message broker type '" + broker + "'. Ignoring...", Tracer.Level.WARNING);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate message broker instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate message broker instances", e);
        }
        return this;
    }

    public final LocalProvisioner launchPlatform() {
        return this.launchAdmin().launchMonitor();
    }

    public final LocalProvisioner startPlatform() {
        return this.startAdmin().startMonitor();
    }

    public final LocalProvisioner stopPlatform() {
        return this.stopMonitor().stopAdmin();
    }

    public final LocalProvisioner terminatePlatform() {
        return this.terminateMonitor().terminateAdmin();
    }

    public final List<String> getPlatformServices() {
        try {
            ArrayList<String> containers = new ArrayList<String>();
            if (this.containerExists(DISCOVERY_CONTAINER_NAME)) {
                containers.add(this.getShortName(DISCOVERY_CONTAINER_NAME));
            }
            if (this.containerExists(AGENT_CONTAINER_NAME)) {
                containers.add(this.getShortName(AGENT_CONTAINER_NAME));
            }
            if (this.containerExists(ADMIN_CONTAINER_NAME)) {
                containers.add(this.getShortName(ADMIN_CONTAINER_NAME));
            }
            if (this.containerExists(MONITOR_CONTAINER_NAME)) {
                containers.add(this.getShortName(MONITOR_CONTAINER_NAME));
            }
            return containers;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get platform services: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get platform services", e);
        }
    }

    public final LocalProvisioner launchServiceContainer(String name, String hostname, int[] servicePorts) {
        try {
            this.tracer.log("Launching service container '" + name + "'.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.createServiceVolume(name);
            this.runServiceContainer(name, hostname, servicePorts);
            this.tracer.log("Service container '" + name + "' launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch service container '" + name + "': " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch service container: " + name, e);
        }
        return this;
    }

    public final LocalProvisioner launchServiceContainer(String name, String hostname) {
        return this.launchServiceContainer(name, hostname, null);
    }

    public final LocalProvisioner runCommandOnServiceContainer(String name, String command) {
        if (name == null) {
            throw new IllegalArgumentException("service container name cannot be null");
        }
        if (command == null) {
            throw new IllegalArgumentException("command cannot be null");
        }
        try {
            String containerName = "rumi-" + name;
            this.tracer.log("Executing command '" + command + "' on service container '" + name + "'.", Tracer.Level.INFO);
            ArrayList<String> dockerCommand = new ArrayList<String>();
            dockerCommand.add("docker");
            dockerCommand.add("exec");
            dockerCommand.add(containerName);
            for (String s : command.split(" ")) {
                dockerCommand.add(s);
            }
            this.executeCommand(null, dockerCommand.toArray(new String[0]));
            this.tracer.log("Successfully run command on service container.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to run command on service container: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to run command on service container", e);
        }
        return this;
    }

    public final LocalProvisioner startServiceContainer(String name) {
        try {
            String containerName = "rumi-" + name;
            this.tracer.log("Starting service container '" + name + "'.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "start", containerName);
            this.tracer.log("Service container '" + name + "' started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start service container '" + name + "': " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start service container: " + name, e);
        }
        return this;
    }

    public final LocalProvisioner stopServiceContainer(String name) {
        try {
            String containerName = "rumi-" + name;
            this.tracer.log("Stopping service container '" + name + "'.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "stop", containerName);
            this.tracer.log("Service container '" + name + "' stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop service container '" + name + "': " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop service container: " + name, e);
        }
        return this;
    }

    public final LocalProvisioner terminateServiceContainer(String name) {
        try {
            String containerName = "rumi-" + name;
            this.tracer.log("Terminating service container '" + name + "'.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "rm", "-f", containerName);
            this.tracer.log("Service container '" + name + "' terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate service container '" + name + "': " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate service container: " + name, e);
        }
        return this;
    }

    public final List<String> getAllServiceContainers() {
        try {
            String[] containerNames;
            ArrayList<String> services = new ArrayList<String>();
            String output = this.executeCommandWithOutput(null, "docker", "ps", "-a", "--filter", "name=^rumi-", "--format", "{{.Names}}");
            for (String containerName : containerNames = output.trim().split("\n")) {
                if (containerName.isEmpty() || !this.isServiceContainer(containerName)) continue;
                services.add(this.getShortName(containerName));
            }
            return services;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get service containers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get service containers", e);
        }
    }

    public final List<String> getRunningServiceContainers() {
        try {
            String[] containerNames;
            ArrayList<String> services = new ArrayList<String>();
            String output = this.executeCommandWithOutput(null, "docker", "ps", "--filter", "name=^rumi-", "--format", "{{.Names}}");
            for (String containerName : containerNames = output.trim().split("\n")) {
                if (containerName.isEmpty() || !this.isServiceContainer(containerName)) continue;
                services.add(this.getShortName(containerName));
            }
            return services;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get running service containers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get running service containers", e);
        }
    }

    public final List<String> getStoppedServiceContainers() {
        try {
            ArrayList<String> services = new ArrayList<String>();
            List<String> allServices = this.getAllServiceContainers();
            List<String> runningServices = this.getRunningServiceContainers();
            for (String service : allServices) {
                if (runningServices.contains(service)) continue;
                services.add(service);
            }
            return services;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get stopped service containers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get stopped service containers", e);
        }
    }

    public final LocalProvisioner startAllServiceContainers() {
        try {
            for (String service : this.getStoppedServiceContainers()) {
                this.startServiceContainer(service);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to start service containers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start service containers", e);
        }
        return this;
    }

    public final LocalProvisioner terminateAllServiceContainers() {
        try {
            for (String service : this.getAllServiceContainers()) {
                this.terminateServiceContainer(service);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate service containers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate service containers", e);
        }
        return this;
    }

    public final LocalProvisioner stopAllServiceContainers() {
        try {
            for (String service : this.getRunningServiceContainers()) {
                this.stopServiceContainer(service);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop service containers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop service containers", e);
        }
        return this;
    }

    public final void provisionEnvironment() {
        this.launchPlatform();
    }

    public final void provisionEnvironmentWithMessageBroker(String messageBroker) {
        if (!(messageBroker == null || messageBroker.equals(SOLACE_SHORT_NAME) || messageBroker.equals(KAFKA_SHORT_NAME) || messageBroker.equals(ACTIVEMQ_SHORT_NAME))) {
            throw new IllegalArgumentException("Unsupported message broker '" + messageBroker + "'");
        }
        this.provisionEnvironment();
        if (messageBroker != null) {
            if (SOLACE_SHORT_NAME.equals(messageBroker)) {
                this.launchSolaceBroker();
            } else if (KAFKA_SHORT_NAME.equals(messageBroker)) {
                this.launchKafkaBroker();
            } else if (ACTIVEMQ_SHORT_NAME.equals(messageBroker)) {
                this.launchActiveMQBroker();
            }
        }
    }

    public final void deploySystem(File xar) {
        if (xar == null) {
            throw new IllegalArgumentException("xar file needs to be specified");
        }
        if (!xar.exists()) {
            throw new IllegalArgumentException("the specified xar file does not exist");
        }
        try {
            this.tracer.log("Deploying the '" + xar.getName() + "' system", Tracer.Level.INFO);
            this.executeCommand(null, "curl", "-X", "POST", "-F", "xar=@" + xar.getAbsolutePath(), "http://localhost:7778/rumi-agent/xars");
            this.tracer.log("System successfully deployed", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to deploy system: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to deploy system", e);
        }
    }

    public final void updateConfig(String section, String key, String value, String filePath) {
        if (section == null) {
            throw new IllegalArgumentException("property section needs to be specified");
        }
        if (key == null) {
            throw new IllegalArgumentException("property key needs to be specified");
        }
        if (filePath == null) {
            throw new IllegalArgumentException("file path needs to be specified");
        }
        try {
            this.tracer.log("Setting configuration property '" + key + "=" + value + "'", Tracer.Level.INFO);
            value = value == null ? "null" : value;
            this.executeCommand(null, "docker", "exec", AGENT_CONTAINER_NAME, "/home/rumi/scripts/update_config.sh", "--section", section, "--key", key, "--value", value, "--file", filePath);
            this.tracer.log("Configuration successfully changed", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to update config: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to update config", e);
        }
    }

    public final void startEnvironment() {
        this.startPlatform();
        this.startAllMessageBrokers();
        this.startAllServiceContainers();
    }

    public final void stopEnvironment() {
        this.stopAllServiceContainers();
        this.stopAllMessageBrokers();
        this.stopPlatform();
    }

    public final void deprovisionEnvironment() {
        this.terminateAllServiceContainers();
        this.terminateAllMessageBrokers();
        this.terminatePlatform();
    }

    public static enum MessageBroker {
        activemq,
        kafka,
        solace;

    }
}

