#!/bin/bash

print_usage() {
    echo "Usage: $0 [--help] [--target <target host>] [--namedaddr <named addr>]" >&2
    exit 1
}

# Parse input parameters
TARGETHOST=""
NAMEDADDR=""
while [[ "$1" == -* ]]; do
    case "$1" in
    --help|-h)
        print_usage
    ;;
    --namedaddr)
        shift
        NAMEDADDR=$1
        shift
    ;;
    --target)
        shift
        TARGETHOST=$1
        shift
    ;;
    *)
        echo "Error: Unknown option: $1" >&2
        exit 1
    ;;
    esac
done

if [[ -z "$TARGETHOST" ]]; then
   print_usage
fi
if [[ -z "$NAMEDADDR" ]]; then
   print_usage
fi

# Wait for TARGETHOST to be reachable
TIMEOUT=300
INTERVAL=5
ELAPSED=0
echo "Waiting for $TARGETHOST to become reachable..."
while ! ping -c 1 "$TARGETHOST" &> /dev/null; do
    if [ "$ELAPSED" -ge "$TIMEOUT" ]; then
        echo "Timeout reached! $TARGETHOST is still not reachable."
        exit 1
    fi

    echo "Still waiting... Retrying in $INTERVAL seconds..."
    sleep "$INTERVAL"
    ELAPSED=$((ELAPSED + INTERVAL))
done
echo "$TARGETHOST is now reachable!"

# Update /etc/resolv.conf on the remote machine
echo "Updating /etc/resolv.conf on $TARGETHOST..."

ssh -i ~/.ssh/rumi rumi@$TARGETHOST <<EOF
    # Ensure /etc/resolv.conf is not immutable
    sudo chattr -i /etc/resolv.conf

    # Backup original resolv.conf if it doesn't already exist
    [ ! -f /etc/resolv.conf.orig ] && sudo cp /etc/resolv.conf /etc/resolv.conf.orig

    # Create a fresh resolv.conf with the required entries
    echo "# Generated by Rumi" | sudo tee /etc/resolv.conf > /dev/null
    echo "search rumi.local" | sudo tee -a /etc/resolv.conf > /dev/null
    echo "nameserver $NAMEDADDR" | sudo tee -a /etc/resolv.conf > /dev/null

    # Lock resolv.conf to prevent overwrites
    sudo chattr +i /etc/resolv.conf

    # Restart NetworkManager using nohup (non-blocking)
    sudo nohup systemctl restart NetworkManager > /dev/null 2>&1 &
EOF

echo "Configuration complete! Original resolv.conf saved as /etc/resolv.conf.orig, and NetworkManager restart triggered."
