/**
 * Copyright 2022 N5 Technologies, Inc
 *
 * This product includes software developed at N5 Technologies, Inc
 * (http://www.n5corp.com/) as well as software licenced to N5 Technologies,
 * Inc under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding
 * copyright ownership.
 *
 * N5 Technologies licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.neeve.tools;

import java.io.File;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import jargs.gnu.CmdLineParser;

import com.neeve.cloud.civo.CivoProvisioner;

/**
 * The 'cloud civo' command handler
 */
final class CivoCommand extends AbstractCommand {
    final private static class ListSSHKeysCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo list-sshkeys <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use]");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            if (apiKey != null) {
                final List<String> keys = CivoProvisioner.create(apiKey).getSSHKeys();
                System.out.println("");
                System.out.println("Provisioned Key Pairs");
                System.out.println("---------------------");
                if (keys.size() > 0) {
                    final Iterator<String> iterator = keys.iterator();
                    while (iterator.hasNext()) {
                        System.out.println("  " + iterator.next());
                    }
                }
                else {
                    System.out.println("  <none>");
                }
                System.out.println("");
            }
            else {
                printUsage();
            }
        }
    }

    final private static class ListDiskImagesCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo list-diskimages <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use]");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            if (apiKey != null) {
                final List<String> images = CivoProvisioner.create(apiKey).getDiskImages();
                System.out.println("");
                System.out.println("Available Images");
                System.out.println("----------------");
                if (images.size() > 0) {
                    final Iterator<String> iterator = images.iterator();
                    while (iterator.hasNext()) {
                        System.out.println("  " + iterator.next());
                    }
                }
                else {
                    System.out.println("  <none>");
                }
                System.out.println("");
            }
            else {
                printUsage();
            }
        }
    }

    final private static class ListInstanceSizesCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo list-instance-sizes <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use]");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            if (apiKey != null) {
                final List<String> sizes = CivoProvisioner.create(apiKey).getInstanceSizes();
                System.out.println("");
                System.out.println("Available Sizes");
                System.out.println("---------------");
                if (sizes.size() > 0) {
                    final Iterator<String> iterator = sizes.iterator();
                    while (iterator.hasNext()) {
                        System.out.println("  " + iterator.next());
                    }
                }
                else {
                    System.out.println("  <none>");
                }
                System.out.println("");
            }
            else {
                printUsage();
            }
        }
    }

    final private static class CreateNetworkCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo create-network <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the network] (required)");
            System.out.println("          [{-n, --name the name to give the new network] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String name = (String)parser.getOptionValue(nameOption, null);
            if (apiKey != null && region != null && name != null) {
                CivoProvisioner.create(apiKey, region).createNetwork(name);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchJumpServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo launch-jump <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            if (apiKey != null && region != null && network != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).launchJumpServer(network, keyPair);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartJumpServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo start-jump <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null) {
                CivoProvisioner.create(apiKey, region).startJumpServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopJumpServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo stop-jump <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null) {
                CivoProvisioner.create(apiKey, region).stopJumpServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateJumpServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo terminate-jump <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).terminateJumpServer(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchAdminServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo launch-admin <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            if (apiKey != null && region != null && network != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).launchAdminServer(network, keyPair);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartAdminServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo start-admin <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null) {
                CivoProvisioner.create(apiKey, region).startAdminServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopAdminServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo stop-admin <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null) {
                CivoProvisioner.create(apiKey, region).stopAdminServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateAdminServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo terminate-admin <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).terminateAdminServer(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchMonitorServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo launch-monitor <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            if (apiKey != null && region != null && network != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).launchMonitorServer(network, keyPair);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartMonitorServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo start-monitor <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null) {
                CivoProvisioner.create(apiKey, region).startMonitorServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopMonitorServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo stop-monitor <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null) {
                CivoProvisioner.create(apiKey, region).stopMonitorServer(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateMonitorServerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo terminate-monitor <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).terminateMonitorServer(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchSolaceBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo launch-solace <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            if (apiKey != null && region != null && network != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).launchSolaceBroker(network, keyPair);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartSolaceBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo start-solace <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null) {
                CivoProvisioner.create(apiKey, region).startSolaceBroker(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopSolaceBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo stop-solace <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null) {
                CivoProvisioner.create(apiKey, region).stopSolaceBroker(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateSolaceBrokerCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo terminate-solace <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).terminateSolaceBroker(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchPlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo launch-jump <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to launch it into] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            if (apiKey != null && region != null && network != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).launchPlatform(network, keyPair);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartPlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo start-jump <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null) {
                CivoProvisioner.create(apiKey, region).startPlatform(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopPlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo stop-jump <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null) {
                CivoProvisioner.create(apiKey, region).stopPlatform(network, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminatePlatformCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo terminate-jump <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to terminate it into] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).terminatePlatform(network, keyPair, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class LaunchServiceInstanceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo launch-instance <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to create the instance] (required)");
            System.out.println("          [{-n, --network the name of the network in which to create the new instance] (required)");
            System.out.println("          [{-k, --keyPair the name of the ssh key to associate with the new instance] (default=rumi)");
            System.out.println("          [{-s, --size the size of the service instance to launch (e.g. g4s.xsmall)] (required)");
            System.out.println("          [{-i, --publicIP whether to assign a public IP to the instance] (default=false)");
            System.out.println("          [{-z, --zone the hosted zone to add this instance's DNS entry to] (default=rumi.local)");
            System.out.println("          [{-a, --name the name of the service instance to launch] (required)");
            System.out.println("          [{-o, --nameInZone the zone name of the service instance to launch] (default=null in which case the instance name is used as the name in the zone)");
            System.out.println("          [{-l, --additionalNamesInZone the additional names to add the zone for this service instance] (default=null in which case no additional names are added to the zone for this instance)");
            System.out.println("          [{-v, --vsize the size, in GiB, of the additional (non-boot) volume on the instance] (default=0)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            final CmdLineParser.Option sizeOption = parser.addStringOption('s', "size");
            final CmdLineParser.Option publicIPOption = parser.addBooleanOption('i', "publicIP");
            final CmdLineParser.Option zoneOption = parser.addStringOption('z', "zone");
            final CmdLineParser.Option nameOption = parser.addStringOption('a', "name");
            final CmdLineParser.Option nameInZoneOption = parser.addStringOption('b', "nameInZone");
            final CmdLineParser.Option additionalNamesInZoneOption = parser.addStringOption('l', "additionalNamesInZone");
            final CmdLineParser.Option vsizeOption = parser.addIntegerOption('v', "vsize");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String size = (String)parser.getOptionValue(sizeOption, null);
            final boolean publicIP = (Boolean)parser.getOptionValue(publicIPOption, false);
            final String zone = (String)parser.getOptionValue(zoneOption, "rumi.local");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String nameInZone = (String)parser.getOptionValue(nameInZoneOption, null);
            final String additionalNamesInZone = (String)parser.getOptionValue(additionalNamesInZoneOption, null);
            final int vsize = (Integer)parser.getOptionValue(vsizeOption, 32);
            if (apiKey != null && region != null && network != null && keyPair != null && name != null && size != null) {
                CivoProvisioner.create(apiKey,region).launchServiceInstance(network, 
                                                                            keyPair, 
                                                                            size, 
                                                                            publicIP, 
                                                                            zone, 
                                                                            name, 
                                                                            nameInZone, 
                                                                            additionalNamesInZone != null ? additionalNamesInZone.split(",") : null,
                                                                            vsize);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartServiceInstanceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo start-instance <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to delete the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to start it into] (required)");
            System.out.println("          [{-a, --name the name of the service instance to start] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nameOption = parser.addStringOption('a', "name");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String name = (String)parser.getOptionValue(nameOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null && name != null) {
                CivoProvisioner.create(apiKey, region).startServiceInstance(network, name, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopServiceInstanceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo stop-instance <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to delete the instance] (required)");
            System.out.println("          [{-n, --network the name of the network to stop it into] (required)");
            System.out.println("          [{-a, --name the name of the service instance to stop] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option nameOption = parser.addStringOption('a', "name");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String name = (String)parser.getOptionValue(nameOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null && name != null) {
                CivoProvisioner.create(apiKey, region).stopServiceInstance(network, name, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class TerminateServiceInstanceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo terminate-instance <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to delete the instance] (required)");
            System.out.println("          [{-n, --network the name of the network in which to delete the instances] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("          [{-i, --name the name of the instance to delete] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            final CmdLineParser.Option nameOption = parser.addStringOption('i', "name");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String name = (String)parser.getOptionValue(nameOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (apiKey != null && region != null && network != null && keyPair != null && name != null) {
                CivoProvisioner.create(apiKey, region).terminateServiceInstance(network, keyPair, name, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class ListInstancesCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo list-instances <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use]");
            System.out.println("          [{-r, --region the Civo region in which to list the instances] (required)");
            System.out.println("          [{-n, --network the name of the network in which to list the instances] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option networkOption = parser.addStringOption('n', "network");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String network = (String)parser.getOptionValue(networkOption, null);
            if (apiKey != null && region != null && network != null) {
                final CivoProvisioner provisioner = CivoProvisioner.create(apiKey, region);
                List<String> instances = provisioner.getPlatformInstances(network);
                System.out.println("");
                System.out.println("Platform Instances");
                System.out.println("------------------");
                if (instances.size() > 0) {
                    final Iterator<String> iterator = instances.iterator();
                    while (iterator.hasNext()) {
                        System.out.println("  " + iterator.next());
                    }
                }
                else {
                    System.out.println("  <none>");
                }
                instances = provisioner.getServiceInstances(network);
                System.out.println("");
                System.out.println("Service Instances");
                System.out.println("-----------------");
                if (instances.size() > 0) {
                    final Iterator<String> iterator = instances.iterator();
                    while (iterator.hasNext()) {
                        System.out.println("  " + iterator.next());
                    }
                }
                else {
                    System.out.println("  <none>");
                }
                System.out.println("");
            }
            else {
                printUsage();
            }
        }
    }

    final private static class ListNetworksCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo list-networks <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use]");
            System.out.println("          [{-r, --region the Civo region in which to list the networks] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            if (apiKey != null && region != null) {
                final List<String> networks = CivoProvisioner.create(apiKey, region).getNetworks();
                System.out.println("");
                System.out.println("Provisioned Networks");
                System.out.println("--------------------");
                if (networks.size() > 0) {
                    final Iterator<String> iterator = networks.iterator();
                    while (iterator.hasNext()) {
                        System.out.println("  " + iterator.next());
                    }
                }
                else {
                    System.out.println("  <none>");
                }
                System.out.println("");
            }
            else {
                printUsage();
            }
        }
    }

    final private static class DeleteNetworkCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo delete-network <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to delete the network] (required)");
            System.out.println("          [{-n, --name the name of the network to delete] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            if (apiKey != null && region != null && name != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).deleteNetwork(name, keyPair);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class ProvisionEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo provision <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to delete the network] (required)");
            System.out.println("          [{-n, --name the name of the network (VPC)] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            if (apiKey != null && region != null && name != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).provisionEnvironment(name, keyPair);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class DeploySystemCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo deploy <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the environment to deploy the system to] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("          [{-x, --xar the xar file] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            final CmdLineParser.Option xarOption = parser.addStringOption('x', "xar");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String xar = (String)parser.getOptionValue(xarOption, null);
            if (apiKey != null && keyPair != null && name != null && xar != null) {
                CivoProvisioner.create(apiKey, region).deploySystem(name, keyPair, new File(xar));
            }
            else {
                printUsage();
            }
        }
    }

    final private static class UpdateEnvironmentConfigCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud aws configure <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the AWS region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the environment whose config is to be updated] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("          [{-s, --section the configuration property section] (required)");
            System.out.println("          [{-e, --key the configuration property key] (required)");
            System.out.println("          [{-v, --value the configuration property value] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            final CmdLineParser.Option sectionOption = parser.addStringOption('s', "section");
            final CmdLineParser.Option keyOption = parser.addStringOption('e', "key");
            final CmdLineParser.Option valueOption = parser.addStringOption('v', "value");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            final String section = (String)parser.getOptionValue(sectionOption, null);
            final String key = (String)parser.getOptionValue(keyOption, null);
            final String value = (String)parser.getOptionValue(valueOption, null);
            if (apiKey != null && keyPair != null && name != null && section != null & key != null && value != null) {
                CivoProvisioner.create(apiKey, region).updateConfig(name, keyPair, section, key, value.equalsIgnoreCase("null") ? null : value);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StartEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo configure <args>' where args are:");
            System.out.println("          [{-p, --profile the Civo profile to use] (required)");
            System.out.println("          [{-r, --region the Civo region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the environment to start] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && name != null) {
                CivoProvisioner.create(profile, region).startEnvironment(name, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class StopEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo configure <args>' where args are:");
            System.out.println("          [{-p, --profile the Civo profile to use] (required)");
            System.out.println("          [{-r, --region the Civo region to use] (default=us-east-1)");
            System.out.println("          [{-n, --name the name of the environment to stop] (required)");
            System.out.println("          [{-o, --nowait whether to not wait for the instance to fully start] (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option profileOption = parser.addStringOption('p', "profile");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option nowaitOption = parser.addBooleanOption('o', "nowait");
            parser.parse(args);
            final String profile = (String)parser.getOptionValue(profileOption, null);
            final String region = (String)parser.getOptionValue(regionOption, "us-east-1");
            final String name = (String)parser.getOptionValue(nameOption, null);
            final boolean nowait = (Boolean)parser.getOptionValue(nowaitOption, false);
            if (profile != null && name != null) {
                CivoProvisioner.create(profile, region).stopEnvironment(name, !nowait);
            }
            else {
                printUsage();
            }
        }
    }

    final private static class DeprovisionEnvironmentCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud civo deprovision <args>' where args are:");
            System.out.println("          [{-y, --api-key the user API key to use] (required)");
            System.out.println("          [{-r, --region the Civo region in which to delete the network] (required)");
            System.out.println("          [{-n, --name the name of the provisioned Rumi environment to delete] (required)");
            System.out.println("          [{-k, --keyPair the network key pair] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "api-key");
            final CmdLineParser.Option regionOption = parser.addStringOption('r', "region");
            final CmdLineParser.Option nameOption = parser.addStringOption('n', "name");
            final CmdLineParser.Option keyPairOption = parser.addStringOption('k', "keyPair");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String region = (String)parser.getOptionValue(regionOption, null);
            final String name = (String)parser.getOptionValue(nameOption, null);
            final String keyPair = (String)parser.getOptionValue(keyPairOption, null);
            if (apiKey != null && region != null && name != null && keyPair != null) {
                CivoProvisioner.create(apiKey, region).deprovisionEnvironment(name, keyPair);
            }
            else {
                printUsage();
            }
        }
    }

    final private static Set<String> commands = Stream.<String>of("list-sshkeys",
                                                                  "list-diskimages",
                                                                  "list-images",
                                                                  "list-instance-sizes",
                                                                  "list-instance-types",
                                                                  "create-network",
                                                                  "launch-jump",
                                                                  "start-jump",
                                                                  "stop-jump",
                                                                  "terminate-jump",
                                                                  "launch-admin",
                                                                  "start-admin",
                                                                  "stop-admin",
                                                                  "terminate-admin",
                                                                  "launch-monitor",
                                                                  "start-monitor",
                                                                  "stop-monitor",
                                                                  "terminate-monitor",
                                                                  "launch-solace",
                                                                  "start-solace",
                                                                  "stop-solace",
                                                                  "terminate-solace",
                                                                  "launch-platform",
                                                                  "start-platform",
                                                                  "stop-platform",
                                                                  "terminate-platform",
                                                                  "launch-instance",
                                                                  "terminate-instance",
                                                                  "list-instances",
                                                                  "list-networks",
                                                                  "delete-network",
                                                                  "provision",
                                                                  "deploy",
                                                                  "configure",
                                                                  "start",
                                                                  "stop",
                                                                  "deprovision").collect(Collectors.toCollection(HashSet::new));

    final private void printTopLevelUsage() {
        System.out.println("");
        System.out.println("      Rumi 'cloud civo' commands:");
        System.out.println("");
        System.out.println("         list-sshkeys                              List the available SSH key pairs");
        System.out.println("         list-diskimages, list-images              List the available disk images (OS images)");
        System.out.println("         list-instance-sizes, list-instance-types  List the available instance sizes");
        System.out.println("         create-network                            Create a new Rumi network");
        System.out.println("         launch-jump                               Provision and start the Rumi Jump Server");
        System.out.println("         start-jump                                Start the Rumi Jump Server");
        System.out.println("         stop-jump                                 Stop the Rumi Jump Server");
        System.out.println("         terminate-jump                            Terminate the Rumi Jump Server");
        System.out.println("         launch-admin                              Provision and start the Rumi Admin Server");
        System.out.println("         start-admin                               Start the Rumi Admin Server");
        System.out.println("         stop-admin                                Stop the Rumi Admin Server");
        System.out.println("         terminate-admin                           Terminate the Rumi Admin Server");
        System.out.println("         launch-monitor                            Provision and start the Rumi Monitor Server");
        System.out.println("         start-monitor                             Start the Rumi Monitor Server");
        System.out.println("         stop-monitor                              Stop the Rumi Monitor Server");
        System.out.println("         terminate-monitor                         Terminate the Rumi Monitor Server");
        System.out.println("         launch-solace                             Provision and start the Rumi Solace Broker");
        System.out.println("         start-solace                              Start the Rumi Solace Broker");
        System.out.println("         stop-solace                               Stop the Rumi Solace Broker");
        System.out.println("         terminate-solace                          Terminate the Rumi Solace Broker");
        System.out.println("         launch-platform                           Provision and start  the Rumi platform components (Jump, Admin, Monitor, Messaging)");
        System.out.println("         start-platform                            Start the Rumi platform components (Jump, Admin, Monitor, Messaging)");
        System.out.println("         stop-platform                             Stop the Rumi platform components (Jump, Admin, Monitor, Messaging)");
        System.out.println("         terminate-platform                        Terminate the Rumi platform components (Jump, Admin, Monitor, Messaging)");
        System.out.println("         launch-instance                           Provision and start a Rumi service instance");
        System.out.println("         terminate-instance                        Terminate the Rumi service instance");
        System.out.println("         list-instances                            List instances provisioned by Rumi");
        System.out.println("         list-networks                             List networks provisioned by Rumi");
        System.out.println("         delete-network                            Delete an existing Rumi network");
        System.out.println("         provision                                 Provision a Rumi environment in the Civo cloud");
        System.out.println("         deploy                                    Deploy a Rumi system to a Rumi environment provisioned in the Civo cloud");
        System.out.println("         configure                                 Update a configuration property of a Rumi environment provisioned in the Civo cloud");
        System.out.println("         start                                     Start all provisioned instances in a Rumi environment in the Civo Cloud");
        System.out.println("         stop                                      Stop all provisioned instances in a Rumi environment in the Civo Cloud");
        System.out.println("         deprovision                               Deprovision an existing Rumi environment in the Civo Cloud");
        System.out.println("");
        System.out.println("      Run 'rumi help cloud civo <command>' for help on a specific command");
        System.out.println("");
    }

    final private static AbstractCommand commandHandler(final String command) {
        if (command.equals("list-sshkeys")) {
            return new ListSSHKeysCommand();
        }
        else if (command.equals("list-diskimages") || command.equals("list-images")) {
            return new ListDiskImagesCommand();
        }
        else if (command.equals("list-instance-sizes") || command.equals("list-instance-types")) {
            return new ListInstanceSizesCommand();
        }
        else if (command.equals("create-network")) {
            return new CreateNetworkCommand();
        }
        else if (command.equals("launch-jump")) {
            return new LaunchJumpServerCommand();
        }
        else if (command.equals("start-jump")) {
            return new StartJumpServerCommand();
        }
        else if (command.equals("stop-jump")) {
            return new StopJumpServerCommand();
        }
        else if (command.equals("terminate-jump")) {
            return new TerminateJumpServerCommand();
        }
        else if (command.equals("launch-admin")) {
            return new LaunchAdminServerCommand();
        }
        else if (command.equals("start-admin")) {
            return new StartAdminServerCommand();
        }
        else if (command.equals("stop-admin")) {
            return new StopAdminServerCommand();
        }
        else if (command.equals("terminate-admin")) {
            return new TerminateAdminServerCommand();
        }
        else if (command.equals("launch-monitor")) {
            return new LaunchMonitorServerCommand();
        }
        else if (command.equals("start-monitor")) {
            return new StartMonitorServerCommand();
        }
        else if (command.equals("stop-monitor")) {
            return new StopMonitorServerCommand();
        }
        else if (command.equals("terminate-monitor")) {
            return new TerminateMonitorServerCommand();
        }
        else if (command.equals("launch-solace")) {
            return new LaunchSolaceBrokerCommand();
        }
        else if (command.equals("start-solace")) {
            return new StartSolaceBrokerCommand();
        }
        else if (command.equals("stop-solace")) {
            return new StopSolaceBrokerCommand();
        }
        else if (command.equals("terminate-solace")) {
            return new TerminateSolaceBrokerCommand();
        }
        else if (command.equals("launch-platform")) {
            return new LaunchPlatformCommand();
        }
        else if (command.equals("start-platform")) {
            return new StartPlatformCommand();
        }
        else if (command.equals("stop-platform")) {
            return new StopPlatformCommand();
        }
        else if (command.equals("terminate-platform")) {
            return new TerminatePlatformCommand();
        }
        else if (command.equals("launch-instance")) {
            return new LaunchServiceInstanceCommand();
        }
        else if (command.equals("start-instance")) {
            return new StartServiceInstanceCommand();
        }
        else if (command.equals("stop-instance")) {
            return new StopServiceInstanceCommand();
        }
        else if (command.equals("terminate-instance")) {
            return new TerminateServiceInstanceCommand();
        }
        else if (command.equals("list-instances")) {
            return new ListInstancesCommand();
        }
        else if (command.equals("list-networks")) {
            return new ListNetworksCommand();
        }
        else if (command.equals("delete-network")) {
            return new DeleteNetworkCommand();
        }
        else if (command.equals("provision")) {
            return new ProvisionEnvironmentCommand();
        }
        else if (command.equals("deploy")) {
            return new DeploySystemCommand();
        }
        else if (command.equals("configure")) {
            return new UpdateEnvironmentConfigCommand();
        }
        else if (command.equals("start")) {
            return new StartEnvironmentCommand();
        }
        else if (command.equals("stop")) {
            return new StopEnvironmentCommand();
        }
        else if (command.equals("deprovision")) {
            return new DeprovisionEnvironmentCommand();
        }
        else {
            throw new IllegalArgumentException("unknown command '" + command + "'");
        }
    }

    @Override
    final void help(final String[] args) {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (commands.contains(command)) {
                commandHandler(command).help(null);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }

    @Override
    final void execute(final String[] args) throws Exception {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (commands.contains(command)) {
                commandHandler(command).execute(args.length > 1 ? Arrays.<String>copyOfRange(args, 1, args.length) : new String[0]);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }
}
