#!/bin/bash

# Usage function
print_usage() {
    echo "Usage: $0 --device <device_path> --mount-point <relative_mount_point> --host <remote_host>"
    echo "Note: mount-point should be relative to /home/rumi (e.g. 'data', not '/data')"
    exit 1
}

# Parse input arguments
DEVICE=""
MOUNT_POINT=""
REMOTE_HOST=""

while [[ "$1" == -* ]]; do
    case "$1" in
    --device)
        shift
        DEVICE=$1
        shift
    ;;
    --mount-point)
        shift
        MOUNT_POINT=$1
        shift
    ;;
    --host)
        shift
        REMOTE_HOST=$1
        shift
    ;;
    *)
        echo "Error: Unknown option: $1" >&2
        exit 1
    ;;
    esac
done

# Validate input parameters
if [[ -z "$DEVICE" || -z "$MOUNT_POINT" || -z "$REMOTE_HOST" ]]; then
    print_usage
fi

# Validate that mount point is not absolute
if [[ "$MOUNT_POINT" = /* ]]; then
    echo "Error: mount-point must be a relative path (e.g. 'data', not '/data')."
    exit 1
fi

# Construct the full path
ABS_MOUNT_POINT="/home/rumi/$MOUNT_POINT"

# Define the remote mount and ownership command
REMOTE_CMD=$(cat <<EOF
    set -e  # Exit script on any error

    # Ensure the device exists
    if [[ ! -b "$DEVICE" ]]; then
        echo "Error: Device $DEVICE does not exist."
        exit 1
    fi

    # Check if the device already has a filesystem
    if sudo blkid "$DEVICE" >/dev/null 2>&1; then
        echo "Warning: Device $DEVICE already has a filesystem."
    else
        # Format the device as XFS
        echo "Formatting $DEVICE as XFS..."
        sudo mkfs -t xfs "$DEVICE"
        echo "Formatting complete."
    fi

    # Ensure the mount point exists
    if [[ ! -d "$ABS_MOUNT_POINT" ]]; then
        echo "Creating mount point directory: $ABS_MOUNT_POINT"
        sudo mkdir -p "$ABS_MOUNT_POINT"
    fi

    # Mount the device if not already mounted
    if mount | grep -q "$ABS_MOUNT_POINT"; then
        echo "Device is already mounted at $ABS_MOUNT_POINT."
    else
        echo "Mounting $DEVICE to $ABS_MOUNT_POINT..."
        sudo mount "$DEVICE" "$ABS_MOUNT_POINT"
    fi

    # Change ownership of /home/rumi recursively to rumi:rumi
    echo "Changing ownership of /home/rumi to rumi:rumi..."
    sudo chown -R rumi:rumi /home/rumi
    echo "Done."
EOF
)

# Execute remotely
ssh -i ~/.ssh/rumi rumi@$REMOTE_HOST "bash -s" <<EOF
    $REMOTE_CMD
EOF

echo "Remote mount and ownership update completed on $REMOTE_HOST."
