#!/bin/bash

print_usage() {
    echo "Usage: $0 [--admin-proxy-port <port>] [--admin-proxy-server <name>] [--discovery-proxy-port <port>] [--discovery-proxy-server <name>]" >&2
    exit 1
}

# Fixed upstream targets
ADMIN_HOST="admin.rumi.local"
ADMIN_HOST_PORT="8001"
DISCOVERY_HOST="discovery.rumi.local"
DISCOVERY_HOST_PORT="8161"

# Proxy-side config (optional)
ADMIN_PROXY_PORT=""
ADMIN_PROXY_SERVER="_"
DISCOVERY_PROXY_PORT=""
DISCOVERY_PROXY_SERVER="_"

# Parse input parameters
while [[ "$1" == -* ]]; do
    case "$1" in
    --admin-proxy-port)
        shift
        ADMIN_PROXY_PORT="$1"
        shift
        ;;
    --admin-proxy-server)
        shift
        ADMIN_PROXY_SERVER="$1"
        shift
        ;;
    --discovery-proxy-port)
        shift
        DISCOVERY_PROXY_PORT="$1"
        shift
        ;;
    --discovery-proxy-server)
        shift
        DISCOVERY_PROXY_SERVER="$1"
        shift
        ;;
    --help|-h)
        print_usage
        ;;
    *)
        echo "Error: Unknown option: $1" >&2
        print_usage
        ;;
    esac
done

# Default proxy listen ports if not set
if [[ -z "$ADMIN_PROXY_PORT" ]]; then
    ADMIN_PROXY_PORT="$ADMIN_HOST_PORT"
fi
if [[ -z "$DISCOVERY_PROXY_PORT" ]]; then
    DISCOVERY_PROXY_PORT="$DISCOVERY_HOST_PORT"
fi

# Add proxy records
/home/rumi/scripts/add_proxy.sh --host-name "$ADMIN_HOST" --host-port "$ADMIN_HOST_PORT" --port "$ADMIN_PROXY_PORT" --server "$ADMIN_PROXY_SERVER"
/home/rumi/scripts/add_proxy.sh --host-name "$DISCOVERY_HOST" --host-port "$DISCOVERY_HOST_PORT" --port "$DISCOVERY_PROXY_PORT" --server "$DISCOVERY_PROXY_SERVER"

# set the DNS resolver on the admin server
/home/rumi/scripts/set_dns_resolver.sh --target "$ADMIN_HOST" --namedaddr "$(hostname -I)"

# Copy private key to admin
MAX_RETRIES=10
RETRY_DELAY=1  # seconds
ATTEMPT=1

while (( ATTEMPT <= MAX_RETRIES )); do
    echo "Attempt #$ATTEMPT: Copying SSH key to $ADMIN_HOST..."

    scp -i ~/.ssh/rumi \
        -o StrictHostKeyChecking=no \
        -o UserKnownHostsFile=/dev/null \
        ~/.ssh/rumi \
        rumi@"$ADMIN_HOST":~/.ssh && {
        echo "✅ Copy succeeded on attempt #$ATTEMPT"
        exit 0
    }

    echo "❌ Attempt #$ATTEMPT failed. Retrying in $RETRY_DELAY second(s)..."
    sleep $RETRY_DELAY
    ((ATTEMPT++))
done

echo "❌ Failed to copy SSH key after $MAX_RETRIES attempts."
exit
