/**
 * Copyright 2022 N5 Technologies, Inc
 *
 * This product includes software developed at N5 Technologies, Inc
 * (http://www.n5corp.com/) as well as software licenced to N5 Technologies,
 * Inc under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding
 * copyright ownership.
 *
 * N5 Technologies licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.neeve.tools;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import jargs.gnu.CmdLineParser;

import com.neeve.cloud.equinix.EquinixProvisioner;

/**
 * The 'cloud equinix' command handler
 */
final class EquinixCommand extends AbstractCommand {
    final private static class GetNetworksCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud equinix get-networks <args>' where args are:");
            System.out.println("          [-k, --apiKey the user API key to use] (required)");
            System.out.println("          [-p, --projectId the UUID of the project whose networks are to be returned] (required)");
            System.out.println("          [-m, --metroCode the code of the metro whose networks are to be returned] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "apiKey");
            final CmdLineParser.Option projectIdOption = parser.addStringOption('p', "projectId");
            final CmdLineParser.Option metroCodeOption = parser.addStringOption('m', "metroCode");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String projectId = (String)parser.getOptionValue(projectIdOption, null);
            final String metroCode = (String)parser.getOptionValue(metroCodeOption, null);
            if (apiKey != null && projectId != null && metroCode != null) {
                final String responseJson = EquinixProvisioner.create().getNetworks(apiKey, projectId, metroCode);
                System.out.println("");
                System.out.println(toPrettyJson(responseJson));
                System.out.println("");
            }
            else {
                printUsage();
            }
        }
    }

    final private static class CreateNetworkCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud equinix create-network <args>' where args are:");
            System.out.println("          [-k, --apiKey the user API key to use] (required)");
            System.out.println("          [-p, --projectId the UUID of the project whose networks are to be returned] (required)");
            System.out.println("          [-m, --metroCode the code of the metro whose networks are to be returned] (required)");
            System.out.println("          [-d, --description a description of the created network] (optional)");
            System.out.println("          [-t, --tags a comma separated set of tags to attach to the network] (optional)");
            System.out.println("");
        }

        final private String[] csvToArray(final String commaSeparatedTags) {
            return (commaSeparatedTags != null)? commaSeparatedTags.split(",") : null;
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "apiKey");
            final CmdLineParser.Option projectIdOption = parser.addStringOption('p', "projectId");
            final CmdLineParser.Option metroCodeOption = parser.addStringOption('m', "metroCode");
            final CmdLineParser.Option descriptionOption = parser.addStringOption('d', "description");
            final CmdLineParser.Option tagsOption = parser.addStringOption('t', "tags");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String projectId = (String)parser.getOptionValue(projectIdOption, null);
            final String metroCode = (String)parser.getOptionValue(metroCodeOption, null);
            final String description = (String)parser.getOptionValue(descriptionOption, null);
            final String commaSeparatedTags = (String)parser.getOptionValue(tagsOption, null);
            if (apiKey != null && projectId != null && metroCode != null) {
                final String responseJson = EquinixProvisioner.create().createNetwork(apiKey, projectId, metroCode, description, csvToArray(commaSeparatedTags));
                System.out.println("");
                System.out.println(toPrettyJson(responseJson));
                System.out.println("");
            }
            else {
                printUsage();
            }
        }
    }

    final private static class DeleteNetworkCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi cloud equinix delete-network <args>' where args are:");
            System.out.println("          [{-k, --apiKey the user API key to use] (required)");
            System.out.println("          [{-i, --id the id of the network to delete] (required)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option apiKeyOption = parser.addStringOption('k', "apiKey");
            final CmdLineParser.Option idOption = parser.addStringOption('i', "id");
            parser.parse(args);
            final String apiKey = (String)parser.getOptionValue(apiKeyOption, null);
            final String id = (String)parser.getOptionValue(idOption, null);
            if (apiKey != null && id != null) {
                final String responseJson = EquinixProvisioner.create().deleteNetwork(apiKey, id);
                System.out.println("");
                System.out.println(toPrettyJson(responseJson));
                System.out.println("");
            }
            else {
                printUsage();
            }
        }
    }

    final private static Set<String> commands = Stream.<String>of("get-networks","create-network","delete-network").collect(Collectors.toCollection(HashSet::new));

    final private void printTopLevelUsage() {
        System.out.println("");
        System.out.println("      Rumi 'cloud equinix' commands:");
        System.out.println("");
        System.out.println("         get-networks     Fetches the list of configured networks");
        System.out.println("         create-network   Create a new network");
        System.out.println("         delete-network   Delete an existing network");
        System.out.println("");
        System.out.println("      Run 'rumi help cloud equinix <command>' for help on a specific command");
        System.out.println("");
    }

    final private static AbstractCommand commandHandler(final String command) {
        if (command.equals("get-networks")) {
            return new GetNetworksCommand();
        }
        if (command.equals("create-network")) {
            return new CreateNetworkCommand();
        }
        if (command.equals("delete-network")) {
            return new DeleteNetworkCommand();
        }
        else {
            throw new IllegalArgumentException("unknown command '" + command + "'");
        }
    }

    @Override
    final void help(final String[] args) {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (commands.contains(command)) {
                commandHandler(command).help(null);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }

    @Override
    final void execute(final String[] args) throws Exception {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (commands.contains(command)) {
                commandHandler(command).execute(args.length > 1 ? Arrays.<String>copyOfRange(args, 1, args.length) : new String[0]);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }
}
