#!/bin/bash

# Usage function
print_usage() {
    echo "Usage: $0 --section <section> --key <key> --value <value> --file <file_path> --host <remote_host>"
    exit 1
}

# Parse input arguments
SECTION=""
KEY=""
VALUE=""
FILE_PATH=""
REMOTE_HOST=""

while [[ "$1" == -* ]]; do
    case "$1" in
    --section)
        shift
        SECTION=$1
        shift
    ;;
    --key)
        shift
        KEY=$1
        shift
    ;;
    --value)
        shift
        VALUE=$1
        shift
    ;;
    --file)
        shift
        FILE_PATH=$1
        shift
    ;;
    --host)
        shift
        REMOTE_HOST=$1
        shift
    ;;
    *)
        echo "Error: Unknown option: $1" >&2
        exit 1
    ;;
    esac
done

# Validate input parameters
if [[ -z "$SECTION" || -z "$KEY" || -z "$VALUE" || -z "$FILE_PATH" || -z "$REMOTE_HOST" ]]; then
    print_usage
fi

# Define the sed script to apply on the remote machine
SED_SCRIPT=$(cat <<EOF
# Ensure the section exists as a commented-out line
if ! grep -q "^# \[$SECTION\]" "$FILE_PATH"; then
    echo -e "\n# [$SECTION]" >> "$FILE_PATH"
fi

# If VALUE is "null", remove the key from the section
if [[ "$VALUE" == "null" ]]; then
    sed -i "/^# \[$SECTION\]/,/^# \[/ {/^$KEY=/d}" "$FILE_PATH"
else
    # Check if the key exists in the section
    if grep -q "^# \[$SECTION\]" "$FILE_PATH"; then
        if grep -A1000 "^# \[$SECTION\]" "$FILE_PATH" | grep -q "^$KEY="; then
            # Update existing key-value pair
            sed -i "/^# \[$SECTION\]/,/^# \[/ {s|^$KEY=.*|$KEY=$VALUE|}" "$FILE_PATH"
        else
            # Insert new key-value pair after the section header
            sed -i "/^# \[$SECTION\]/a $KEY=$VALUE" "$FILE_PATH"
        fi
    fi
fi
EOF
)

# Execute the command remotely via SSH
ssh -i ~/.ssh/rumi rumi@$REMOTE_HOST "bash -s" <<EOF
    $SED_SCRIPT
EOF

echo "Configuration updated successfully on $REMOTE_HOST."
