/**
 * Copyright 2022 N5 Technologies, Inc
 *
 * This product includes software developed at N5 Technologies, Inc
 * (http://www.n5corp.com/) as well as software licenced to N5 Technologies,
 * Inc under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding
 * copyright ownership.
 *
 * N5 Technologies licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.neeve.tools;

import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Handler for the 'rumi tools' command.
 *
 * This parent command provides access to analysis and troubleshooting tools:
 * - tlt: Transaction Log Tool for querying transaction logs
 * - hbdump: Heartbeats dump tool for analyzing heartbeat data
 */
final class ToolsCommand extends AbstractCommand {

    final private static Set<String> topLevelCommands = Stream.<String>of("tlt","hbdump").collect(Collectors.toCollection(HashSet::new));

    final private void printTopLevelUsage() {
        System.out.println("Usage: rumi tools <command> [options]");
        System.out.println();
        System.out.println("Analysis and troubleshooting tools for Rumi applications");
        System.out.println();
        System.out.println("Available commands:");
        System.out.println("  tlt        Transaction Log Tool - query and analyze transaction logs");
        System.out.println("  hbdump     Heartbeats dump tool - analyze heartbeat data");
        System.out.println();
        System.out.println("Run 'rumi tools <command> --help' for detailed information on a command.");
    }

    final private static AbstractCommand commandHandler(final String command) {
        switch(command.toLowerCase()) {
            case "tlt":
                return new TltCommand();
            case "hbdump":
                return new HeartbeatsDumpCommand();
            default:
                return null;
        }
    }

    @Override
    final void help(final String[] args) {
        if (args.length == 0) {
            printTopLevelUsage();
        }
        else {
            final AbstractCommand commandHandler = commandHandler(args[0]);
            if (commandHandler == null) {
                System.err.println("Unknown tools command '" + args[0] + "'");
                System.err.println();
                printTopLevelUsage();
            }
            else {
                final String[] subArgs = new String[args.length - 1];
                System.arraycopy(args, 1, subArgs, 0, subArgs.length);
                commandHandler.help(subArgs);
            }
        }
    }

    @Override
    final void execute(final String[] args) throws Exception {
        if (args.length == 0) {
            printTopLevelUsage();
            System.exit(1);
        }
        else if (!topLevelCommands.contains(args[0].toLowerCase())) {
            System.err.println("Unknown tools command '" + args[0] + "'");
            System.err.println();
            printTopLevelUsage();
            System.exit(1);
        }
        else {
            final AbstractCommand commandHandler = commandHandler(args[0]);
            final String[] subArgs = new String[args.length - 1];
            System.arraycopy(args, 1, subArgs, 0, subArgs.length);
            commandHandler.execute(subArgs);
        }
    }
}
