/**
 * Copyright 2022 N5 Technologies, Inc
 *
 * This product includes software developed at N5 Technologies, Inc
 * (http://www.n5corp.com/) as well as software licenced to N5 Technologies,
 * Inc under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding
 * copyright ownership.
 *
 * N5 Technologies licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.neeve.tools;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * The Rumi CLI main entry point
 */
public class CliMain {
    final private static Set<String> topLevelCommands = Stream.<String>of("help","quickstart","cloud","tools").collect(Collectors.toCollection(HashSet::new));

    final private static void printTopLevelUsage() {
        System.out.println("");
        System.out.println("      Rumi -- The \"No-DB\" Platform");
        System.out.println("");
        System.out.println("      Rumi CLI is Rumi's command line tool. Try:");
        System.out.println("");
        System.out.println("      rumi help commands        list the top level commands");
        System.out.println("      rumi help <command>       help on a specific command");
        System.out.println("");
    }

    final private static void printTopLevelCommands() {
        System.out.println("");
        System.out.println("      Rumi CLI commands:");
        System.out.println("");
        System.out.println("         quickstart Commands to generate Rumi quickstart applications");
        System.out.println("         cloud      Commands related to operations that provision Rumi in IaaS cloud providers");
        System.out.println("         tools      Analysis and troubleshooting tools for Rumi applications");
        System.out.println("");
        System.out.println("      Run 'rumi help <command>' for help on a specific command");
        System.out.println("");
    }

    final private static AbstractCommand commandHandler(final String command) {
        if (command.equals("quickstart")) {
            return new QuickstartCommand();
        }
        else if (command.equals("cloud")) {
            return new CloudCommand();
        }
        else if (command.equals("tools")) {
            return new ToolsCommand();
        }
        else {
            throw new IllegalArgumentException("unknown command '" + command + "'");
        }
    }

    public static void main(String args[]) throws Exception {
        if (args.length > 0) {
            final String topLevelCommand = args[0].toLowerCase();
            if (topLevelCommands.contains(topLevelCommand)) {
                if (topLevelCommand.equals("help")) {
                    if (args.length > 1) {
                        final String command = args[1].toLowerCase();
                        if (command.equals("commands")) {
                            printTopLevelCommands();
                        }
                        else if (topLevelCommands.contains(command)) {
                            commandHandler(command).help(args.length > 2 ? Arrays.<String>copyOfRange(args, 2, args.length) : new String[0]);
                        }
                        else {
                            System.out.println("No help for '" + command + "'.");
                        }
                    }
                    else {
                        printTopLevelUsage();
                    }
                }
                else {
                    try {
                        commandHandler(topLevelCommand).execute(args.length > 1 ? Arrays.<String>copyOfRange(args, 1, args.length) : new String[0]);
                    }
                    catch (Throwable e) {
                        e.printStackTrace();
                        System.out.println("");
                        System.out.println("**** " + (e.getMessage() != null ? e.getMessage() : e.toString()));
                        System.out.println("");
                    }
                }
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }
}
