/*
 * Decompiled with CFR 0.152.
 */
package com.neeve.cloud.local;

import com.neeve.trace.Tracer;
import java.io.BufferedReader;
import java.io.File;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.List;

public final class LocalProvisioner {
    private static final String NETWORK_NAME = "rumi";
    private static final String DISCOVERY_SHORT_NAME = "discovery";
    private static final String DISCOVERY_CONTAINER_NAME = "rumi-discovery";
    private static final String ADMIN_COMPOSE_URL = "https://downloads.n5corp.com/rumi/admin/latest/docker-compose.yml";
    private static final String[] ADMIN_VOLUMES = new String[]{"rumi-influxdb-data", "rumi-discovery-data", "rumi-agent-data", "rumi-admin-data"};
    private static final String ADMIN_SHORT_NAME = "admin";
    private static final String ADMIN_CONTAINER_NAME = "rumi-admin";
    private static final String ADMIN_SCRIPTS_DIR = "/home/rumi/scripts";
    private static final String AGENT_SHORT_NAME = "agent";
    private static final String AGENT_CONTAINER_NAME = "rumi-agent";
    private static final String MONITOR_SHORT_NAME = "monitor";
    private static final String MONITOR_CONTAINER_NAME = "rumi-monitor";
    private static final String MONITOR_VOLUME = "rumi-monitor-data";
    private static final String MONITOR_IMAGE = "neeve/nvx-rumi-monitor:latest";
    private static final String ACTIVEMQ_SHORT_NAME = "activemq";
    private static final String ACTIVEMQ_CONTAINER_NAME = "rumi-activemq";
    private static final String ACTIVEMQ_VOLUME = "rumi-activemq-data";
    private static final String ACTIVEMQ_IMAGE = "apache/activemq-classic:5.18.6";
    private static final String KAFKA_SHORT_NAME = "kafka";
    private static final String KAFKA_CONTAINER_NAME = "rumi-kafka";
    private static final String KAFKA_VOLUME = "rumi-kafka-data";
    private static final String KAFKA_IMAGE = "apache/kafka:latest";
    private static final String SOLACE_SHORT_NAME = "solace";
    private static final String SOLACE_CONTAINER_NAME = "rumi-solace";
    private static final String SERVICE_IMAGE = "neeve/nvx-rumi-worker:latest";
    private static final String RUMI_CONTROLLER_CONF_FILE = "/home/rumi/rumi-agent/data/controller/controller.conf";
    private final Tracer tracer = Tracer.get((String)"nv.cloud");
    private File adminDockerComposeFile;
    private String dockerComposeCommand;

    private LocalProvisioner() {
        this.detectDockerCompose();
    }

    public static LocalProvisioner create() {
        return new LocalProvisioner();
    }

    private void detectDockerCompose() {
        int exitCode;
        Process process;
        ProcessBuilder pb;
        try {
            pb = new ProcessBuilder("docker", "compose", "version");
            pb.redirectErrorStream(true);
            process = pb.start();
            exitCode = process.waitFor();
            if (exitCode == 0) {
                this.dockerComposeCommand = "docker compose";
                if (this.tracer.debug) {
                    this.tracer.log("Using 'docker compose' command", Tracer.Level.DEBUG);
                }
                return;
            }
        }
        catch (Exception pb2) {
            // empty catch block
        }
        try {
            pb = new ProcessBuilder("docker-compose", "version");
            pb.redirectErrorStream(true);
            process = pb.start();
            exitCode = process.waitFor();
            if (exitCode == 0) {
                this.dockerComposeCommand = "docker-compose";
                if (this.tracer.debug) {
                    this.tracer.log("Using 'docker-compose' command", Tracer.Level.DEBUG);
                }
                return;
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        throw new RuntimeException("Neither 'docker compose' nor 'docker-compose' command found. Please install Docker Compose.");
    }

    private void ensureDockerRunning() throws Exception {
        if (this.tracer.debug) {
            this.tracer.log("Checking if Docker is running...", Tracer.Level.DEBUG);
        }
        this.executeCommand(null, "docker", "info");
        if (this.tracer.debug) {
            this.tracer.log("Docker is running.", Tracer.Level.DEBUG);
        }
    }

    private void createDockerNetwork() throws Exception {
        this.tracer.log("...creating docker network 'rumi'.", Tracer.Level.VERBOSE);
        try {
            this.executeCommand(null, "docker", "network", "create", NETWORK_NAME);
        }
        catch (Exception e) {
            String output = this.executeCommandWithOutput(null, "docker", "network", "ls", "--filter", "name=rumi", "--format", "{{.Name}}");
            if (!output.contains(NETWORK_NAME)) {
                throw e;
            }
            this.tracer.log("...network already exists.", Tracer.Level.VERBOSE);
        }
    }

    private void createAdminDockerVolumes() throws Exception {
        for (String volume : ADMIN_VOLUMES) {
            this.tracer.log("...creating docker volume '" + volume + "'.", Tracer.Level.VERBOSE);
            try {
                this.executeCommand(null, "docker", "volume", "create", volume);
            }
            catch (Exception e) {
                String output = this.executeCommandWithOutput(null, "docker", "volume", "ls", "--filter", "name=" + volume, "--format", "{{.Name}}");
                if (!output.contains(volume)) {
                    throw e;
                }
                this.tracer.log("...volume already exists.", Tracer.Level.VERBOSE);
            }
        }
    }

    private File getAdminDockerComposeFile() {
        File rumiDir = new File(System.getProperty("user.home"), ".rumi");
        if (!rumiDir.exists()) {
            rumiDir.mkdirs();
        }
        return new File(rumiDir, "admin-docker-compose.yml");
    }

    private void downloadAdminDockerCompose() throws Exception {
        if (this.tracer.debug) {
            this.tracer.log("...downloading admin docker-compose.yml.", Tracer.Level.DEBUG);
        }
        this.adminDockerComposeFile = this.getAdminDockerComposeFile();
        URL url = new URL(ADMIN_COMPOSE_URL);
        Files.copy(url.openStream(), this.adminDockerComposeFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        if (this.tracer.debug) {
            this.tracer.log("Downloaded latest admin docker-compose.yml from: https://downloads.n5corp.com/rumi/admin/latest/docker-compose.yml", Tracer.Level.DEBUG);
        }
    }

    private void runAdminDockerCompose() throws Exception {
        this.tracer.log("...running admin docker-compose up.", Tracer.Level.VERBOSE);
        if (this.dockerComposeCommand.equals("docker compose")) {
            this.executeCommand(null, "docker", "compose", "-f", this.adminDockerComposeFile.getAbsolutePath(), "up", "-d");
        } else {
            this.executeCommand(null, "docker-compose", "-f", this.adminDockerComposeFile.getAbsolutePath(), "up", "-d");
        }
    }

    private void waitForContainer(String containerName) throws Exception {
        this.tracer.log("...waiting for container " + containerName + " to be ready.", Tracer.Level.VERBOSE);
        int maxRetries = 30;
        int retryCount = 0;
        boolean ready = false;
        while (retryCount < maxRetries && !ready) {
            try {
                this.executeCommand(null, "docker", "exec", containerName, "echo", "ready");
                ready = true;
            }
            catch (Exception e) {
                if (++retryCount < maxRetries) {
                    Thread.sleep(1000L);
                    continue;
                }
                throw new RuntimeException("Container " + containerName + " did not become ready in time", e);
            }
        }
        if (this.tracer.debug) {
            this.tracer.log("Container " + containerName + " is ready.", Tracer.Level.DEBUG);
        }
    }

    private void copyScriptsToAdminContainers() throws Exception {
        this.tracer.log("...copying scripts to admin containers.", Tracer.Level.VERBOSE);
        this.waitForContainer(AGENT_CONTAINER_NAME);
        this.executeCommand(null, "docker", "exec", AGENT_CONTAINER_NAME, "mkdir", "-p", ADMIN_SCRIPTS_DIR);
        String scriptsResourcePath = "/scripts/docker/admin";
        File tempScriptsDir = new File(System.getProperty("java.io.tmpdir"), "rumi-scripts");
        if (!tempScriptsDir.exists()) {
            tempScriptsDir.mkdirs();
        }
        File runAdminScriptFile = new File(tempScriptsDir, "run_admin_script.sh");
        try (InputStream is = this.getClass().getResourceAsStream(scriptsResourcePath + "/run_admin_script.sh");){
            if (is == null) {
                throw new RuntimeException("Resource not found: " + scriptsResourcePath + "/run_admin_script.sh");
            }
            Files.copy(is, runAdminScriptFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        this.executeCommand(null, "docker", "cp", runAdminScriptFile.getAbsolutePath(), "rumi-agent:/home/rumi/scripts/run_admin_script.sh");
        this.executeCommand(null, "docker", "exec", "--user", "root", AGENT_CONTAINER_NAME, "chown", "rumi:rumi", "/home/rumi/scripts/run_admin_script.sh");
        this.executeCommand(null, "docker", "exec", "--user", "root", AGENT_CONTAINER_NAME, "chmod", "+x", "/home/rumi/scripts/run_admin_script.sh");
        File updateConfigFile = new File(tempScriptsDir, "update_config.sh");
        try (InputStream is = this.getClass().getResourceAsStream(scriptsResourcePath + "/update_config.sh");){
            if (is == null) {
                throw new RuntimeException("Resource not found: " + scriptsResourcePath + "/update_config.sh");
            }
            Files.copy(is, updateConfigFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        this.executeCommand(null, "docker", "cp", updateConfigFile.getAbsolutePath(), "rumi-agent:/home/rumi/scripts/update_config.sh");
        this.executeCommand(null, "docker", "exec", "--user", "root", AGENT_CONTAINER_NAME, "chown", "rumi:rumi", "/home/rumi/scripts/update_config.sh");
        this.executeCommand(null, "docker", "exec", "--user", "root", AGENT_CONTAINER_NAME, "chmod", "+x", "/home/rumi/scripts/update_config.sh");
        runAdminScriptFile.delete();
        updateConfigFile.delete();
    }

    private void createMonitorVolume() throws Exception {
        this.tracer.log("...creating docker volume 'rumi-monitor-data'.", Tracer.Level.VERBOSE);
        try {
            this.executeCommand(null, "docker", "volume", "create", MONITOR_VOLUME);
        }
        catch (Exception e) {
            String output = this.executeCommandWithOutput(null, "docker", "volume", "ls", "--filter", "name=rumi-monitor-data", "--format", "{{.Name}}");
            if (!output.contains(MONITOR_VOLUME)) {
                throw e;
            }
            this.tracer.log("...volume already exists.", Tracer.Level.VERBOSE);
        }
    }

    private void runMonitorContainer() throws Exception {
        this.tracer.log("...running monitor container.", Tracer.Level.VERBOSE);
        this.executeCommand(null, "docker", "run", "-d", "--name", MONITOR_CONTAINER_NAME, "--hostname", "monitor.rumi.local", "--network", NETWORK_NAME, "-p", "3000:3000", "-v", "rumi-monitor-data:/var/lib/grafana", "--restart", "unless-stopped", "--health-cmd", "curl -f http://localhost:3000/api/health", "--health-interval", "30s", "--health-timeout", "10s", "--health-retries", "3", "--health-start-period", "30s", MONITOR_IMAGE);
    }

    private void createActiveMQVolume() throws Exception {
        this.tracer.log("...creating docker volume 'rumi-activemq-data'.", Tracer.Level.VERBOSE);
        try {
            this.executeCommand(null, "docker", "volume", "create", ACTIVEMQ_VOLUME);
        }
        catch (Exception e) {
            String output = this.executeCommandWithOutput(null, "docker", "volume", "ls", "--filter", "name=rumi-activemq-data", "--format", "{{.Name}}");
            if (!output.contains(ACTIVEMQ_VOLUME)) {
                throw e;
            }
            this.tracer.log("...volume already exists.", Tracer.Level.VERBOSE);
        }
    }

    private void runActiveMQContainer() throws Exception {
        this.tracer.log("...running activemq broker container.", Tracer.Level.VERBOSE);
        this.executeCommand(null, "docker", "run", "-d", "--name", ACTIVEMQ_CONTAINER_NAME, "--hostname", "activemq.rumi.local", "--network", NETWORK_NAME, "-p", "61617:61616", "-p", "8162:8161", "-v", "rumi-activemq-data:/opt/apache-activemq/data", "--restart", "unless-stopped", "--health-cmd", "curl -f -u admin:admin http://localhost:8161", "--health-interval", "30s", "--health-timeout", "10s", "--health-retries", "5", "--health-start-period", "30s", ACTIVEMQ_IMAGE);
    }

    private void createKafkaVolume() throws Exception {
        this.tracer.log("...creating docker volume 'rumi-kafka-data'.", Tracer.Level.VERBOSE);
        try {
            this.executeCommand(null, "docker", "volume", "create", KAFKA_VOLUME);
        }
        catch (Exception e) {
            String output = this.executeCommandWithOutput(null, "docker", "volume", "ls", "--filter", "name=rumi-kafka-data", "--format", "{{.Name}}");
            if (!output.contains(KAFKA_VOLUME)) {
                throw e;
            }
            this.tracer.log("...volume already exists.", Tracer.Level.VERBOSE);
        }
    }

    private void runKafkaContainer() throws Exception {
        this.tracer.log("...running kafka broker container.", Tracer.Level.VERBOSE);
        this.executeCommand(null, "docker", "run", "-d", "--name", KAFKA_CONTAINER_NAME, "--hostname", "kafka.rumi.local", "--network", NETWORK_NAME, "-p", "9092:9092", "-v", "rumi-kafka-data:/var/lib/kafka/data", "--restart", "unless-stopped", "-e", "KAFKA_NODE_ID=1", "-e", "KAFKA_PROCESS_ROLES=broker,controller", "-e", "KAFKA_LISTENERS=PLAINTEXT://0.0.0.0:9092,CONTROLLER://0.0.0.0:9093", "-e", "KAFKA_ADVERTISED_LISTENERS=PLAINTEXT://kafka.rumi.local:9092", "-e", "KAFKA_CONTROLLER_LISTENER_NAMES=CONTROLLER", "-e", "KAFKA_LISTENER_SECURITY_PROTOCOL_MAP=CONTROLLER:PLAINTEXT,PLAINTEXT:PLAINTEXT", "-e", "KAFKA_CONTROLLER_QUORUM_VOTERS=1@kafka.rumi.local:9093", "-e", "KAFKA_OFFSETS_TOPIC_REPLICATION_FACTOR=1", "-e", "KAFKA_TRANSACTION_STATE_LOG_REPLICATION_FACTOR=1", "-e", "KAFKA_TRANSACTION_STATE_LOG_MIN_ISR=1", KAFKA_IMAGE);
    }

    private void runSolaceContainer() throws Exception {
        this.tracer.log("...running solace broker container.", Tracer.Level.VERBOSE);
        String volumeName = "rumi-solace-data";
        try {
            this.executeCommand(null, "docker", "volume", "create", "rumi-solace-data");
        }
        catch (Exception exception) {
            // empty catch block
        }
        this.executeCommand(null, "docker", "run", "-d", "--name", SOLACE_CONTAINER_NAME, "--hostname", "solace.rumi.local", "--network", NETWORK_NAME, "-p", "8008:8008", "-p", "1443:1443", "-p", "1943:1943", "-p", "1883:1883", "-p", "2222:2222", "-p", "5671:5671", "-p", "5672:5672", "-p", "8000:8000", "-p", "8080:8080", "-p", "8443:8443", "-p", "8883:8883", "-p", "9000:9000", "-p", "9443:9443", "-p", "55003:55003", "-p", "55443:55443", "-p", "55554:55555", "-v", "rumi-solace-data:/var/lib/solace", "--shm-size", "1g", "--ulimit", "nofile=2448:1048576", "--ulimit", "core=-1", "--restart", "unless-stopped", "-e", "username_admin_globalaccesslevel=admin", "-e", "username_admin_password=admin", "solace/solace-pubsub-standard:latest");
    }

    private void deleteDirectory(File directory) {
        if (directory.exists()) {
            File[] files = directory.listFiles();
            if (files != null) {
                for (File file : files) {
                    if (file.isDirectory()) {
                        this.deleteDirectory(file);
                        continue;
                    }
                    file.delete();
                }
            }
            directory.delete();
        }
    }

    private void createServiceVolume(String name) throws Exception {
        String volumeName = "rumi-" + name + "-data";
        this.tracer.log("...creating docker volume '" + volumeName + "'.", Tracer.Level.VERBOSE);
        try {
            this.executeCommand(null, "docker", "volume", "create", volumeName);
        }
        catch (Exception e) {
            String output = this.executeCommandWithOutput(null, "docker", "volume", "ls", "--filter", "name=" + volumeName, "--format", "{{.Name}}");
            if (!output.contains(volumeName)) {
                throw e;
            }
            this.tracer.log("...volume already exists.", Tracer.Level.VERBOSE);
        }
    }

    private void runServiceInstance(String name, String hostname, String[] hostnameAliases, int[] servicePorts) throws Exception {
        String containerName = "rumi-" + name;
        String fullHostname = hostname + ".rumi.local";
        String volumeName = "rumi-" + name + "-data";
        this.tracer.log("...running service instance '" + name + "'.", Tracer.Level.VERBOSE);
        ArrayList<String> command = new ArrayList<String>();
        command.add("docker");
        command.add("run");
        command.add("-d");
        command.add("--name");
        command.add(containerName);
        command.add("--hostname");
        command.add(fullHostname);
        command.add("--network");
        command.add(NETWORK_NAME);
        if (hostnameAliases != null) {
            for (String alias : hostnameAliases) {
                command.add("--network-alias");
                command.add(alias + ".rumi.local");
            }
        }
        command.add("-v");
        command.add(volumeName + ":/home/rumi");
        if (servicePorts != null) {
            for (int port : servicePorts) {
                command.add("-p");
                command.add(port + ":" + port);
            }
        }
        command.add("--restart");
        command.add("unless-stopped");
        command.add(SERVICE_IMAGE);
        this.executeCommand(null, command.toArray(new String[0]));
    }

    private boolean isServiceInstance(String containerName) {
        return !containerName.equals(DISCOVERY_CONTAINER_NAME) && !containerName.equals(AGENT_CONTAINER_NAME) && !containerName.equals(ADMIN_CONTAINER_NAME) && !containerName.equals(MONITOR_CONTAINER_NAME) && !containerName.equals(ACTIVEMQ_CONTAINER_NAME) && !containerName.equals(KAFKA_CONTAINER_NAME) && !containerName.equals(SOLACE_CONTAINER_NAME);
    }

    private String getShortName(String containerName) {
        String prefix = "rumi-";
        if (containerName.startsWith("rumi-")) {
            return containerName.substring("rumi-".length());
        }
        return containerName;
    }

    private boolean containerExists(String containerName) throws Exception {
        try {
            String output = this.executeCommandWithOutput(null, "docker", "ps", "-a", "--filter", "name=^" + containerName + "$", "--format", "{{.Names}}");
            return output.trim().equals(containerName);
        }
        catch (Exception e) {
            return false;
        }
    }

    private boolean containerIsRunning(String containerName) throws Exception {
        try {
            String output = this.executeCommandWithOutput(null, "docker", "ps", "--filter", "name=^" + containerName + "$", "--format", "{{.Names}}");
            return output.trim().equals(containerName);
        }
        catch (Exception e) {
            return false;
        }
    }

    private void executeCommand(File workingDir, String ... command) throws Exception {
        String line;
        ProcessBuilder pb = new ProcessBuilder(command);
        if (workingDir != null) {
            pb.directory(workingDir);
        }
        pb.redirectErrorStream(true);
        Process process = pb.start();
        BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
        while ((line = reader.readLine()) != null) {
            if (!this.tracer.debug) continue;
            this.tracer.log(line, Tracer.Level.DEBUG);
        }
        int exitCode = process.waitFor();
        if (exitCode != 0) {
            throw new RuntimeException("Command failed with exit code " + exitCode + ": " + String.join((CharSequence)" ", command));
        }
    }

    private String executeCommandWithOutput(File workingDir, String ... command) throws Exception {
        String line;
        ProcessBuilder pb = new ProcessBuilder(command);
        if (workingDir != null) {
            pb.directory(workingDir);
        }
        pb.redirectErrorStream(true);
        Process process = pb.start();
        StringBuilder output = new StringBuilder();
        BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
        while ((line = reader.readLine()) != null) {
            output.append(line).append("\n");
        }
        int exitCode = process.waitFor();
        if (exitCode != 0) {
            throw new RuntimeException("Command failed with exit code " + exitCode + ": " + String.join((CharSequence)" ", command));
        }
        return output.toString();
    }

    public final LocalProvisioner launchAdmin() {
        try {
            this.tracer.log("Launching Rumi admin.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.createAdminDockerVolumes();
            this.downloadAdminDockerCompose();
            this.runAdminDockerCompose();
            this.copyScriptsToAdminContainers();
            this.tracer.log("Rumi admin launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch Rumi admin: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch admin", e);
        }
        return this;
    }

    public final LocalProvisioner startAdmin() {
        try {
            this.tracer.log("Starting Rumi admin.", Tracer.Level.INFO);
            this.downloadAdminDockerCompose();
            this.tracer.log("...starting admin containers.", Tracer.Level.VERBOSE);
            if (this.dockerComposeCommand.equals("docker compose")) {
                this.executeCommand(null, "docker", "compose", "-f", this.adminDockerComposeFile.getAbsolutePath(), "start");
            } else {
                this.executeCommand(null, "docker-compose", "-f", this.adminDockerComposeFile.getAbsolutePath(), "start");
            }
            this.tracer.log("Rumi admin started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start Rumi admin: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start admin", e);
        }
        return this;
    }

    public final LocalProvisioner runAdminScript(String system, String script, String params) {
        if (system == null) {
            throw new IllegalArgumentException("system name cannot be null");
        }
        if (script == null) {
            throw new IllegalArgumentException("script name cannot be null");
        }
        try {
            this.tracer.log("Executing admin script '" + script + "' on the '" + system + "' system with '" + params + "'.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "exec", AGENT_CONTAINER_NAME, "/home/rumi/scripts/run_admin_script.sh", "--system", system, "--script", script, "--param", params == null ? "dummy=value" : params);
            this.tracer.log("Successfully run admin script.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to run admin script: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to run admin script", e);
        }
        return this;
    }

    public final LocalProvisioner stopAdmin() {
        try {
            this.tracer.log("Stopping Rumi admin.", Tracer.Level.INFO);
            this.downloadAdminDockerCompose();
            this.tracer.log("...stopping admin containers.", Tracer.Level.VERBOSE);
            if (this.dockerComposeCommand.equals("docker compose")) {
                this.executeCommand(null, "docker", "compose", "-f", this.adminDockerComposeFile.getAbsolutePath(), "stop");
            } else {
                this.executeCommand(null, "docker-compose", "-f", this.adminDockerComposeFile.getAbsolutePath(), "stop");
            }
            this.tracer.log("Rumi admin stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop Rumi admin: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop admin", e);
        }
        return this;
    }

    public final LocalProvisioner terminateAdmin() {
        try {
            this.tracer.log("Terminating Rumi admin.", Tracer.Level.INFO);
            this.downloadAdminDockerCompose();
            this.tracer.log("...stopping and removing admin containers.", Tracer.Level.VERBOSE);
            if (this.dockerComposeCommand.equals("docker compose")) {
                this.executeCommand(null, "docker", "compose", "-f", this.adminDockerComposeFile.getAbsolutePath(), "down");
            } else {
                this.executeCommand(null, "docker-compose", "-f", this.adminDockerComposeFile.getAbsolutePath(), "down");
            }
            this.tracer.log("...deleting admin volumes.", Tracer.Level.VERBOSE);
            for (String volume : ADMIN_VOLUMES) {
                try {
                    this.executeCommand(null, "docker", "volume", "rm", "-f", volume);
                }
                catch (Exception e) {
                    this.tracer.log("Failed to remove volume " + volume + ": " + e.getMessage(), Tracer.Level.WARNING);
                }
            }
            this.adminDockerComposeFile.delete();
            this.tracer.log("Rumi admin terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate Rumi admin: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate admin", e);
        }
        return this;
    }

    public final LocalProvisioner launchMonitor() {
        try {
            this.tracer.log("Launching the Rumi monitor.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.createMonitorVolume();
            this.runMonitorContainer();
            this.tracer.log("Rumi monitor launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch Rumi monitor: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch monitor", e);
        }
        return this;
    }

    public final LocalProvisioner startMonitor() {
        try {
            this.tracer.log("Starting the Rumi monitor.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "start", MONITOR_CONTAINER_NAME);
            this.tracer.log("Rumi monitor started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start Rumi monitor: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start monitor", e);
        }
        return this;
    }

    public final LocalProvisioner stopMonitor() {
        try {
            this.tracer.log("Stopping the Rumi monitor.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "stop", MONITOR_CONTAINER_NAME);
            this.tracer.log("Rumi monitor stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop Rumi monitor: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop monitor", e);
        }
        return this;
    }

    public final LocalProvisioner terminateMonitor() {
        try {
            this.tracer.log("Terminating the Rumi monitor.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "rm", "-f", MONITOR_CONTAINER_NAME);
            try {
                this.executeCommand(null, "docker", "volume", "rm", "-f", MONITOR_VOLUME);
            }
            catch (Exception e) {
                this.tracer.log("Failed to remove volume rumi-monitor-data: " + e.getMessage(), Tracer.Level.WARNING);
            }
            this.tracer.log("Rumi monitor terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate Rumi monitor: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate monitor", e);
        }
        return this;
    }

    public final LocalProvisioner launchSolaceBroker() {
        try {
            this.tracer.log("Launching solace.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.runSolaceContainer();
            this.tracer.log("Solace launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch solace: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch solace", e);
        }
        return this;
    }

    public final LocalProvisioner startSolaceBroker() {
        try {
            this.tracer.log("Starting solace.", Tracer.Level.INFO);
            this.tracer.log("...starting solace container.", Tracer.Level.VERBOSE);
            this.executeCommand(null, "docker", "start", SOLACE_CONTAINER_NAME);
            this.tracer.log("Solace started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start solace: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start solace", e);
        }
        return this;
    }

    public final LocalProvisioner stopSolaceBroker() {
        try {
            this.tracer.log("Stopping solace.", Tracer.Level.INFO);
            this.tracer.log("...stopping solace container.", Tracer.Level.VERBOSE);
            this.executeCommand(null, "docker", "stop", SOLACE_CONTAINER_NAME);
            this.tracer.log("Solace stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop solace: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop solace", e);
        }
        return this;
    }

    public final LocalProvisioner terminateSolaceBroker() {
        try {
            this.tracer.log("Terminating solace.", Tracer.Level.INFO);
            this.tracer.log("...stopping and removing solace container.", Tracer.Level.VERBOSE);
            this.executeCommand(null, "docker", "rm", "-f", SOLACE_CONTAINER_NAME);
            String volumeName = "rumi-solace-data";
            this.tracer.log("...deleting solace volume.", Tracer.Level.VERBOSE);
            try {
                this.executeCommand(null, "docker", "volume", "rm", "-f", "rumi-solace-data");
            }
            catch (Exception e) {
                this.tracer.log("Failed to remove volume rumi-solace-data: " + e.getMessage(), Tracer.Level.WARNING);
            }
            this.tracer.log("Solace terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate solace: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate solace", e);
        }
        return this;
    }

    public final LocalProvisioner launchKafkaBroker() {
        try {
            this.tracer.log("Launching kafka.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.createKafkaVolume();
            this.runKafkaContainer();
            this.tracer.log("Kafka launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch kafka: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch kafka", e);
        }
        return this;
    }

    public final LocalProvisioner startKafkaBroker() {
        try {
            this.tracer.log("Starting kafka.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "start", KAFKA_CONTAINER_NAME);
            this.tracer.log("Kafka started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start kafka: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start kafka", e);
        }
        return this;
    }

    public final LocalProvisioner stopKafkaBroker() {
        try {
            this.tracer.log("Stopping kafka.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "stop", KAFKA_CONTAINER_NAME);
            this.tracer.log("Kafka stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop kafka: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop kafka", e);
        }
        return this;
    }

    public final LocalProvisioner terminateKafkaBroker() {
        try {
            this.tracer.log("Terminating kafka.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "rm", "-f", KAFKA_CONTAINER_NAME);
            try {
                this.executeCommand(null, "docker", "volume", "rm", "-f", KAFKA_VOLUME);
            }
            catch (Exception e) {
                this.tracer.log("Failed to remove volume rumi-kafka-data: " + e.getMessage(), Tracer.Level.WARNING);
            }
            this.tracer.log("Kafka terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate kafka: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate kafka", e);
        }
        return this;
    }

    public final LocalProvisioner launchActiveMQBroker() {
        try {
            this.tracer.log("Launching activemq.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.createActiveMQVolume();
            this.runActiveMQContainer();
            this.tracer.log("ActiveMQ launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch activemq: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch activemq", e);
        }
        return this;
    }

    public final LocalProvisioner startActiveMQBroker() {
        try {
            this.tracer.log("Starting activemq.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "start", ACTIVEMQ_CONTAINER_NAME);
            this.tracer.log("ActiveMQ started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start activemq: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start activemq", e);
        }
        return this;
    }

    public final LocalProvisioner stopActiveMQBroker() {
        try {
            this.tracer.log("Stopping activemq.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "stop", ACTIVEMQ_CONTAINER_NAME);
            this.tracer.log("ActiveMQ stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop activemq: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop activemq", e);
        }
        return this;
    }

    public final LocalProvisioner terminateActiveMQBroker() {
        try {
            this.tracer.log("Terminating activemq.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "rm", "-f", ACTIVEMQ_CONTAINER_NAME);
            try {
                this.executeCommand(null, "docker", "volume", "rm", "-f", ACTIVEMQ_VOLUME);
            }
            catch (Exception e) {
                this.tracer.log("Failed to remove volume rumi-activemq-data: " + e.getMessage(), Tracer.Level.WARNING);
            }
            this.tracer.log("ActiveMQ terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate activemq: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate activemq", e);
        }
        return this;
    }

    public final List<String> getAllMessageBrokers() {
        try {
            ArrayList<String> brokers = new ArrayList<String>();
            if (this.containerExists(ACTIVEMQ_CONTAINER_NAME)) {
                brokers.add(this.getShortName(ACTIVEMQ_CONTAINER_NAME));
            }
            if (this.containerExists(KAFKA_CONTAINER_NAME)) {
                brokers.add(this.getShortName(KAFKA_CONTAINER_NAME));
            }
            if (this.containerExists(SOLACE_CONTAINER_NAME)) {
                brokers.add(this.getShortName(SOLACE_CONTAINER_NAME));
            }
            return brokers;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get message brokers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get message brokers", e);
        }
    }

    public final List<String> getRunningMessageBrokers() {
        try {
            ArrayList<String> brokers = new ArrayList<String>();
            if (this.containerIsRunning(ACTIVEMQ_CONTAINER_NAME)) {
                brokers.add(this.getShortName(ACTIVEMQ_CONTAINER_NAME));
            }
            if (this.containerIsRunning(KAFKA_CONTAINER_NAME)) {
                brokers.add(this.getShortName(KAFKA_CONTAINER_NAME));
            }
            if (this.containerIsRunning(SOLACE_CONTAINER_NAME)) {
                brokers.add(this.getShortName(SOLACE_CONTAINER_NAME));
            }
            return brokers;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get running message brokers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get running message brokers", e);
        }
    }

    public final List<String> getStoppedMessageBrokers() {
        try {
            ArrayList<String> brokers = new ArrayList<String>();
            if (this.containerExists(ACTIVEMQ_CONTAINER_NAME) && !this.containerIsRunning(ACTIVEMQ_CONTAINER_NAME)) {
                brokers.add(this.getShortName(ACTIVEMQ_CONTAINER_NAME));
            }
            if (this.containerExists(KAFKA_CONTAINER_NAME) && !this.containerIsRunning(KAFKA_CONTAINER_NAME)) {
                brokers.add(this.getShortName(KAFKA_CONTAINER_NAME));
            }
            if (this.containerExists(SOLACE_CONTAINER_NAME) && !this.containerIsRunning(SOLACE_CONTAINER_NAME)) {
                brokers.add(this.getShortName(SOLACE_CONTAINER_NAME));
            }
            return brokers;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get stopped message brokers: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get stopped message brokers", e);
        }
    }

    public final LocalProvisioner startAllMessageBrokers() {
        try {
            for (String broker : this.getStoppedMessageBrokers()) {
                if (ACTIVEMQ_SHORT_NAME.equals(broker)) {
                    this.startActiveMQBroker();
                    continue;
                }
                if (KAFKA_SHORT_NAME.equals(broker)) {
                    this.startKafkaBroker();
                    continue;
                }
                if (SOLACE_SHORT_NAME.equals(broker)) {
                    this.startSolaceBroker();
                    continue;
                }
                this.tracer.log("Encountered unknown message broker type '" + broker + "'. Ignoring...", Tracer.Level.WARNING);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to start message broker instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start message broker instances", e);
        }
        return this;
    }

    public final LocalProvisioner stopAllMessageBrokers() {
        try {
            for (String broker : this.getRunningMessageBrokers()) {
                if (ACTIVEMQ_SHORT_NAME.equals(broker)) {
                    this.stopActiveMQBroker();
                    continue;
                }
                if (KAFKA_SHORT_NAME.equals(broker)) {
                    this.stopKafkaBroker();
                    continue;
                }
                if (SOLACE_SHORT_NAME.equals(broker)) {
                    this.stopSolaceBroker();
                    continue;
                }
                this.tracer.log("Encountered unknown message broker type '" + broker + "'. Ignoring...", Tracer.Level.WARNING);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop message broker instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop message broker instances", e);
        }
        return this;
    }

    public final LocalProvisioner terminateAllMessageBrokers() {
        try {
            for (String broker : this.getAllMessageBrokers()) {
                if (ACTIVEMQ_SHORT_NAME.equals(broker)) {
                    this.terminateActiveMQBroker();
                    continue;
                }
                if (KAFKA_SHORT_NAME.equals(broker)) {
                    this.terminateKafkaBroker();
                    continue;
                }
                if (SOLACE_SHORT_NAME.equals(broker)) {
                    this.terminateSolaceBroker();
                    continue;
                }
                this.tracer.log("Encountered unknown message broker type '" + broker + "'. Ignoring...", Tracer.Level.WARNING);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate message broker instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate message broker instances", e);
        }
        return this;
    }

    public final LocalProvisioner launchPlatform() {
        return this.launchAdmin().launchMonitor();
    }

    public final LocalProvisioner startPlatform() {
        return this.startAdmin().startMonitor();
    }

    public final LocalProvisioner stopPlatform() {
        return this.stopMonitor().stopAdmin();
    }

    public final LocalProvisioner terminatePlatform() {
        return this.terminateMonitor().terminateAdmin();
    }

    public final List<String> getPlatformServices() {
        try {
            ArrayList<String> containers = new ArrayList<String>();
            if (this.containerExists(DISCOVERY_CONTAINER_NAME)) {
                containers.add(this.getShortName(DISCOVERY_CONTAINER_NAME));
            }
            if (this.containerExists(AGENT_CONTAINER_NAME)) {
                containers.add(this.getShortName(AGENT_CONTAINER_NAME));
            }
            if (this.containerExists(ADMIN_CONTAINER_NAME)) {
                containers.add(this.getShortName(ADMIN_CONTAINER_NAME));
            }
            if (this.containerExists(MONITOR_CONTAINER_NAME)) {
                containers.add(this.getShortName(MONITOR_CONTAINER_NAME));
            }
            return containers;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get platform services: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get platform services", e);
        }
    }

    public final LocalProvisioner launchServiceInstance(String name, String hostname, String[] hostnameAliases, int[] servicePorts) {
        try {
            this.tracer.log("Launching service instance '" + name + "'.", Tracer.Level.INFO);
            this.ensureDockerRunning();
            this.createDockerNetwork();
            this.createServiceVolume(name);
            this.runServiceInstance(name, hostname, hostnameAliases, servicePorts);
            this.tracer.log("Service instance '" + name + "' launched successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to launch service instance '" + name + "': " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to launch service instance: " + name, e);
        }
        return this;
    }

    public final LocalProvisioner launchServiceInstance(String name, String hostname, int[] servicePorts) {
        return this.launchServiceInstance(name, hostname, null, servicePorts);
    }

    public final LocalProvisioner launchServiceInstance(String name, String hostname, String[] hostnameAliases) {
        return this.launchServiceInstance(name, hostname, hostnameAliases, null);
    }

    public final LocalProvisioner launchServiceInstance(String name, String hostname) {
        return this.launchServiceInstance(name, hostname, null, null);
    }

    public final LocalProvisioner runCommandOnServiceInstance(String name, String command) {
        if (name == null) {
            throw new IllegalArgumentException("service instance name cannot be null");
        }
        if (command == null) {
            throw new IllegalArgumentException("command cannot be null");
        }
        try {
            String containerName = "rumi-" + name;
            this.tracer.log("Executing command '" + command + "' on service instance '" + name + "'.", Tracer.Level.INFO);
            ArrayList<String> dockerCommand = new ArrayList<String>();
            dockerCommand.add("docker");
            dockerCommand.add("exec");
            dockerCommand.add(containerName);
            for (String s : command.split(" ")) {
                dockerCommand.add(s);
            }
            this.executeCommand(null, dockerCommand.toArray(new String[0]));
            this.tracer.log("Successfully run command on service instance.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to run command on service instance: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to run command on service instance", e);
        }
        return this;
    }

    public final LocalProvisioner startServiceInstance(String name) {
        try {
            String containerName = "rumi-" + name;
            this.tracer.log("Starting service instance '" + name + "'.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "start", containerName);
            this.tracer.log("Service instance '" + name + "' started successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to start service instance '" + name + "': " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start service instance: " + name, e);
        }
        return this;
    }

    public final LocalProvisioner stopServiceInstance(String name) {
        try {
            String containerName = "rumi-" + name;
            this.tracer.log("Stopping service instance '" + name + "'.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "stop", containerName);
            this.tracer.log("Service instance '" + name + "' stopped successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop service instance '" + name + "': " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop service instance: " + name, e);
        }
        return this;
    }

    public final LocalProvisioner terminateServiceInstance(String name) {
        try {
            String containerName = "rumi-" + name;
            String volumeName = "rumi-" + name + "-data";
            this.tracer.log("Terminating service instance '" + name + "'.", Tracer.Level.INFO);
            this.executeCommand(null, "docker", "rm", "-f", containerName);
            this.tracer.log("Deleting volume '" + volumeName + "'.", Tracer.Level.VERBOSE);
            this.executeCommand(null, "docker", "volume", "rm", volumeName);
            this.tracer.log("Service instance '" + name + "' terminated successfully.", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate service instance '" + name + "': " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate service instance: " + name, e);
        }
        return this;
    }

    public final List<String> getAllServiceInstances() {
        try {
            String[] containerNames;
            ArrayList<String> services = new ArrayList<String>();
            String output = this.executeCommandWithOutput(null, "docker", "ps", "-a", "--filter", "name=^rumi-", "--format", "{{.Names}}");
            for (String containerName : containerNames = output.trim().split("\n")) {
                if (containerName.isEmpty() || !this.isServiceInstance(containerName)) continue;
                services.add(this.getShortName(containerName));
            }
            return services;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get service instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get service instances", e);
        }
    }

    public final List<String> getRunningServiceInstances() {
        try {
            String[] containerNames;
            ArrayList<String> services = new ArrayList<String>();
            String output = this.executeCommandWithOutput(null, "docker", "ps", "--filter", "name=^rumi-", "--format", "{{.Names}}");
            for (String containerName : containerNames = output.trim().split("\n")) {
                if (containerName.isEmpty() || !this.isServiceInstance(containerName)) continue;
                services.add(this.getShortName(containerName));
            }
            return services;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get running service instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get running service instances", e);
        }
    }

    public final List<String> getStoppedServiceInstances() {
        try {
            ArrayList<String> services = new ArrayList<String>();
            List<String> allServices = this.getAllServiceInstances();
            List<String> runningServices = this.getRunningServiceInstances();
            for (String service : allServices) {
                if (runningServices.contains(service)) continue;
                services.add(service);
            }
            return services;
        }
        catch (Exception e) {
            this.tracer.log("Failed to get stopped service instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to get stopped service instances", e);
        }
    }

    public final LocalProvisioner startAllServiceInstances() {
        try {
            for (String service : this.getStoppedServiceInstances()) {
                this.startServiceInstance(service);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to start service instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to start service instances", e);
        }
        return this;
    }

    public final LocalProvisioner terminateAllServiceInstances() {
        try {
            for (String service : this.getAllServiceInstances()) {
                this.terminateServiceInstance(service);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to terminate service instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to terminate service instances", e);
        }
        return this;
    }

    public final LocalProvisioner stopAllServiceInstances() {
        try {
            for (String service : this.getRunningServiceInstances()) {
                this.stopServiceInstance(service);
            }
        }
        catch (Exception e) {
            this.tracer.log("Failed to stop service instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to stop service instances", e);
        }
        return this;
    }

    public final void provisionEnvironment() {
        this.launchPlatform();
    }

    public final void provisionEnvironmentWithMessageBroker(String messageBroker) {
        if (!(messageBroker == null || messageBroker.equals(SOLACE_SHORT_NAME) || messageBroker.equals(KAFKA_SHORT_NAME) || messageBroker.equals(ACTIVEMQ_SHORT_NAME))) {
            throw new IllegalArgumentException("Unsupported message broker '" + messageBroker + "'");
        }
        this.provisionEnvironment();
        if (messageBroker != null) {
            if (SOLACE_SHORT_NAME.equals(messageBroker)) {
                this.launchSolaceBroker();
            } else if (KAFKA_SHORT_NAME.equals(messageBroker)) {
                this.launchKafkaBroker();
            } else if (ACTIVEMQ_SHORT_NAME.equals(messageBroker)) {
                this.launchActiveMQBroker();
            }
        }
    }

    public final void deploySystem(File xar) {
        if (xar == null) {
            throw new IllegalArgumentException("xar file needs to be specified");
        }
        if (!xar.exists()) {
            throw new IllegalArgumentException("the specified xar file does not exist");
        }
        try {
            this.tracer.log("Deploying the '" + xar.getName() + "' system", Tracer.Level.INFO);
            this.executeCommand(null, "curl", "-X", "POST", "-F", "xar=@" + xar.getAbsolutePath(), "http://localhost:7770/rumi-agent/xars");
            this.tracer.log("System successfully deployed", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to deploy system: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to deploy system", e);
        }
    }

    public final void updateConfig(String section, String key, String value) {
        if (section == null) {
            throw new IllegalArgumentException("property section needs to be specified");
        }
        if (key == null) {
            throw new IllegalArgumentException("property key needs to be specified");
        }
        try {
            this.tracer.log("Setting configuration property '" + key + "=" + value + "'", Tracer.Level.INFO);
            value = value == null ? "null" : value;
            this.executeCommand(null, "docker", "exec", AGENT_CONTAINER_NAME, "/home/rumi/scripts/update_config.sh", "--section", section, "--key", key, "--value", value, "--file", RUMI_CONTROLLER_CONF_FILE);
            this.tracer.log("Configuration successfully changed", Tracer.Level.INFO);
        }
        catch (Exception e) {
            this.tracer.log("Failed to update config: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to update config", e);
        }
    }

    public final void listInstances() {
        try {
            String[] lines;
            this.tracer.log("Listing all containers in the local Rumi environment.", Tracer.Level.INFO);
            String output = this.executeCommandWithOutput(null, "docker", "ps", "-a", "--filter", "name=rumi-", "--format", "{{.Names}}\t{{.Status}}\t{{.Image}}");
            System.out.println("");
            System.out.println("Platform Containers");
            System.out.println("-------------------");
            boolean foundPlatform = false;
            for (String line : lines = output.split("\n")) {
                String[] parts;
                String name;
                if (line.trim().isEmpty() || !(name = (parts = line.split("\t"))[0]).contains("-admin") && !name.contains("-agent") && !name.contains("-monitor")) continue;
                System.out.println("  " + line);
                foundPlatform = true;
            }
            if (!foundPlatform) {
                System.out.println("  <none>");
            }
            System.out.println("");
            System.out.println("Message Broker Containers");
            System.out.println("-------------------------");
            boolean foundBroker = false;
            for (String line : lines) {
                String[] parts;
                String name;
                if (line.trim().isEmpty() || !(name = (parts = line.split("\t"))[0]).contains("-activemq") && !name.contains("-kafka") && !name.contains("-solace")) continue;
                System.out.println("  " + line);
                foundBroker = true;
            }
            if (!foundBroker) {
                System.out.println("  <none>");
            }
            System.out.println("");
            System.out.println("Service Containers");
            System.out.println("------------------");
            boolean foundService = false;
            for (String line : lines) {
                String[] parts;
                String name;
                if (line.trim().isEmpty() || (name = (parts = line.split("\t"))[0]).contains("-admin") || name.contains("-agent") || name.contains("-monitor") || name.contains("-activemq") || name.contains("-kafka") || name.contains("-solace")) continue;
                System.out.println("  " + line);
                foundService = true;
            }
            if (!foundService) {
                System.out.println("  <none>");
            }
            System.out.println("");
        }
        catch (Exception e) {
            this.tracer.log("Failed to list instances: " + e.getMessage(), Tracer.Level.SEVERE);
            throw new RuntimeException("Failed to list instances", e);
        }
    }

    public final void startEnvironment() {
        this.startPlatform();
        this.startAllMessageBrokers();
        this.startAllServiceInstances();
    }

    public final void stopEnvironment() {
        this.stopAllServiceInstances();
        this.stopAllMessageBrokers();
        this.stopPlatform();
    }

    public final void deprovisionEnvironment() {
        this.terminateAllServiceInstances();
        this.terminateAllMessageBrokers();
        this.terminatePlatform();
    }

    public static enum MessageBroker {
        activemq,
        kafka,
        solace;

    }
}

