/**
 * Copyright 2022 N5 Technologies, Inc
 *
 * This product includes software developed at N5 Technologies, Inc
 * (http://www.n5corp.com/) as well as software licenced to N5 Technologies,
 * Inc under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding
 * copyright ownership.
 *
 * N5 Technologies licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.neeve.tools;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import jargs.gnu.CmdLineParser;

import com.neeve.appbuilder.ApplicationBuilder;
import com.neeve.appbuilder.ApplicationBuilder.AppParams;
import com.neeve.appbuilder.ApplicationBuilder.BuildTool;
import com.neeve.appbuilder.ApplicationBuilder.EncodingType;
import com.neeve.appbuilder.ApplicationBuilder.MessagingProvider;
import com.neeve.appbuilder.ServiceBuilder;
import com.neeve.appbuilder.ServiceBuilder.ServiceParams;
import com.neeve.appbuilder.ServiceBuilder.ServiceType;
import com.neeve.appbuilder.ServiceBuilder.ServiceHAModel;

/**
 * The 'quickstart' top level command handler
 */
final class QuickstartCommand extends AbstractCommand {
    final private static class GenerateQuickstartAppCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi quickstart app <args>' where args are:");
            System.out.println("          [{-e, --encoding-type] the app's encoding type (default=quark)");
            System.out.println("          [{-m, --messaging-provider] the app's messaging provider (default=activemq)");
            System.out.println("          [{-t, --target-dir] the target directory (default=.)");
            System.out.println("          [{-s, --only-scaffolding] whether to only create the app scaffolding or the full app (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option encodingTypeOption = parser.addStringOption('e', "encoding-type");
            final CmdLineParser.Option messagingProviderOption = parser.addStringOption('m', "messaging-provider");
            final CmdLineParser.Option targetDirOption = parser.addStringOption('t', "target-dir");
            final CmdLineParser.Option onlyScaffoldingOption = parser.addBooleanOption('s', "only-scaffolding");
            parser.parse(args);
            final String encodingTypeStr = (String)parser.getOptionValue(encodingTypeOption, "quark");
            final String messagingProviderStr = (String)parser.getOptionValue(messagingProviderOption, "activemq");
            final String targetDirStr = (String)parser.getOptionValue(targetDirOption, ".");
            final boolean onlyScaffolding = (Boolean)parser.getOptionValue(onlyScaffoldingOption, false);
            final AppParams appParams = new AppParams("Quickstart",
                                                      targetDirStr,
                                                      "org.rumi.quickstart",
                                                      "org.rumi",
                                                      "rumi",
                                                      "4.0.584",
                                                      "4.0.39",
                                                      "2.0.29",
                                                      EncodingType.fromString(encodingTypeStr),
                                                      MessagingProvider.fromString(messagingProviderStr),
                                                      BuildTool.MAVEN);
            final String appRoot = appParams.getAppRoot();
            if (onlyScaffolding) {
                new ApplicationBuilder().createApplication(appParams);
            }
            else {
                new ApplicationBuilder().createApplication(appParams)
                                        .newServiceBuilder()
                                        .createService(new ServiceParams(appRoot, "Processor", ServiceType.PROCESSOR, ServiceHAModel.STATE_REPLICATION, true, 1))
                                        .createService(new ServiceParams(appRoot, "CSVWriter", ServiceType.CSVWRITER, null, false, 1))
                                        .createService(new ServiceParams(appRoot, "Driver", ServiceType.DRIVER, null, false, 1));
            }
        }
    }

    final private static class GenerateQuickstartProcessorServiceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi quickstart processor' where args are:");
            System.out.println("          [{-a, --app-root] root of the application to add the processor service (default=.)");
            System.out.println("          [{-s, --non-clustered] whether the processor is created non-clustered (default=false)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option appRootOption = parser.addStringOption('r', "app-root");
            final CmdLineParser.Option standaloneOption = parser.addBooleanOption('s', "non-clustered");
            parser.parse(args);
            final String appRoot = (String)parser.getOptionValue(appRootOption, "rumi-quickstart");
            final boolean standalone = (Boolean)parser.getOptionValue(standaloneOption, false);
            new ServiceBuilder().createService(new ServiceParams(appRoot, "Processor", ServiceType.PROCESSOR, ServiceHAModel.STATE_REPLICATION, !standalone, 1));
        }
    }

    final private static class GenerateQuickstartDriverServiceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi quickstart driver' where args are:");
            System.out.println("          [{-a, --app-root] root of the application to add the driver service (default=.)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option appRootOption = parser.addStringOption('r', "app-root");
            parser.parse(args);
            final String appRoot = (String)parser.getOptionValue(appRootOption, "rumi-quickstart");
            new ServiceBuilder().createService(new ServiceParams(appRoot, "Driver", ServiceType.DRIVER, null, false, 1));
        }
    }

    final private static class GenerateQuickstartCSVWriterServiceCommand extends AbstractCommand {

        final private static void printUsage() {
            System.out.println("");
            System.out.println("      'rumi quickstart csvwriter' where args are:");
            System.out.println("          [{-a, --app-root] root of the application to add the CSV Writer connector service (default=.)");
            System.out.println("");
        }

        @Override
        final void help(final String[] args) {
            printUsage();
        }

        @Override
        final void execute(final String[] args) throws Exception {
            final CmdLineParser parser = new CmdLineParser();
            final CmdLineParser.Option appRootOption = parser.addStringOption('r', "app-root");
            parser.parse(args);
            final String appRoot = (String)parser.getOptionValue(appRootOption, "rumi-quickstart");
            new ServiceBuilder().createService(new ServiceParams(appRoot, "CSVWriter", ServiceType.CSVWRITER, null, false, 1));
        }
    }

    final private static Set<String> topLevelCommands = Stream.<String>of("app","processor","driver","csvwriter").collect(Collectors.toCollection(HashSet::new));

    final private void printTopLevelUsage() {
        System.out.println("");
        System.out.println("      Rumi 'quickstart' commands:");
        System.out.println("");                                                                             
        System.out.println("         app        Generate the Rumi quickstart application");
        System.out.println("         processor  Generate the Rumi quickstart processor service");
        System.out.println("         driver     Generate the Rumi quickstart driver service");
        System.out.println("         csvwriter  Generate the Rumi quickstart CSV writer service");
        System.out.println("");
        System.out.println("      Run 'rumi help quickstart <command>' for help on a specific command");
        System.out.println("");
    }

    final private static AbstractCommand commandHandler(final String command) {
        if (command.equals("app")) {
            return new GenerateQuickstartAppCommand();
        }
        else if (command.equals("processor")) {
            return new GenerateQuickstartProcessorServiceCommand();
        }
        else if (command.equals("driver")) {
            return new GenerateQuickstartDriverServiceCommand();
        }
        else if (command.equals("csvwriter")) {
            return new GenerateQuickstartCSVWriterServiceCommand();
        }
        else {
            throw new IllegalArgumentException("unknown command '" + command + "'");
        }
    }

    @Override
    final void help(final String[] args) {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (topLevelCommands.contains(command)) {
                commandHandler(command).help(args.length > 1 ? Arrays.<String>copyOfRange(args, 1, args.length) : new String[0]);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }

    @Override
    final void execute(final String[] args) throws Exception {
        if (args.length > 0) {
            final String command = args[0].toLowerCase();
            if (topLevelCommands.contains(command)) {
                commandHandler(command).execute(args.length > 1 ? Arrays.<String>copyOfRange(args, 1, args.length) : new String[0]);
            }
            else {
                printTopLevelUsage();
            }
        }
        else {
            printTopLevelUsage();
        }
    }
}
