/*
 * Decompiled with CFR 0.152.
 */
package com.adobe.acs.commons.packagegarbagecollector;

import java.io.IOException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Period;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import javax.annotation.Nonnull;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import org.apache.jackrabbit.vault.packaging.JcrPackage;
import org.apache.jackrabbit.vault.packaging.JcrPackageDefinition;
import org.apache.jackrabbit.vault.packaging.JcrPackageManager;
import org.apache.jackrabbit.vault.packaging.PackageId;
import org.apache.jackrabbit.vault.packaging.Packaging;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.apache.sling.event.jobs.Job;
import org.apache.sling.event.jobs.consumer.JobConsumer;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(service={JobConsumer.class}, immediate=true, property={"job.topics=com/adobe/acs/commons/PackageGarbageCollectionJob"})
public class PackageGarbageCollectionJob
implements JobConsumer {
    public static final DateTimeFormatter LOCALIZED_DATE_FORMATTER = DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM);
    private static final Logger LOG = LoggerFactory.getLogger(PackageGarbageCollectionJob.class);
    private static final String SERVICE_USER = "package-garbage-collection";
    @Reference
    Packaging packaging;
    @Reference
    ResourceResolverFactory resourceResolverFactory;

    public JobConsumer.JobResult process(Job job) {
        String groupName = (String)job.getProperty("groupName", String.class);
        Integer maxAgeInDays = (Integer)job.getProperty("maxAgeInDays", Integer.class);
        boolean removeNotInstalledPackages = (Boolean)job.getProperty("removeNotInstalledPackages", (Object)false);
        int packagesRemoved = 0;
        LOG.debug("Job Configuration: [Group Name: {}, Service User: {}, Age of Package {} days,Remove not installed packages: {}]", new Object[]{groupName, SERVICE_USER, maxAgeInDays, removeNotInstalledPackages});
        try (ResourceResolver resourceResolver = this.resourceResolverFactory.getServiceResourceResolver(Collections.singletonMap("sling.service.subservice", SERVICE_USER));){
            Session session = (Session)resourceResolver.adaptTo(Session.class);
            JcrPackageManager packageManager = this.packaging.getPackageManager(session);
            List packages = packageManager.listPackages(groupName, false);
            for (JcrPackage jcrPackage : packages) {
                String packageDescription = this.getPackageDescription(jcrPackage);
                LOG.info("Processing package {}", (Object)packageDescription);
                if (this.isPackageOldEnough(jcrPackage, maxAgeInDays)) {
                    if (removeNotInstalledPackages && !this.isInstalled(jcrPackage)) {
                        packageManager.remove(jcrPackage);
                        ++packagesRemoved;
                        LOG.info("Deleted not-installed package {}", (Object)packageDescription);
                        continue;
                    }
                    if (this.isInstalled(jcrPackage) && !this.isLatestInstalled(jcrPackage, packageManager.listPackages(groupName, false))) {
                        packageManager.remove(jcrPackage);
                        ++packagesRemoved;
                        LOG.info("Deleted installed package {} since it is not the latest installed version.", (Object)packageDescription);
                        continue;
                    }
                    LOG.info("Not removing package because it's the current installed one {}", (Object)packageDescription);
                    continue;
                }
                LOG.debug("Not removing package because it's not old enough {}", (Object)packageDescription);
            }
        }
        catch (IOException | RepositoryException | LoginException e) {
            if (packagesRemoved > 0) {
                LOG.error("Package Garbage Collector job partially failed - Removed {} packages", (Object)packagesRemoved);
            }
            LOG.error("Unable to finish clearing packages", e);
            return JobConsumer.JobResult.FAILED;
        }
        LOG.info("Package Garbage Collector job finished - Removed {} packages", (Object)packagesRemoved);
        return JobConsumer.JobResult.OK;
    }

    private boolean isInstalled(JcrPackage jcrPackage) {
        PackageDefinition definition = new PackageDefinition(jcrPackage);
        return definition.getLastUnpacked() != null;
    }

    private boolean isLatestInstalled(JcrPackage jcrPackage, List<JcrPackage> installedPackages) {
        Optional<JcrPackage> lastInstalledPackageOptional = installedPackages.stream().filter(installedPackage -> {
            PackageDefinition definition = new PackageDefinition((JcrPackage)installedPackage);
            return definition.isSameNameAndGroup(jcrPackage);
        }).filter(pkg -> new PackageDefinition((JcrPackage)pkg).getLastUnpacked() != null).max(Comparator.comparing(pkg -> new PackageDefinition((JcrPackage)pkg).getLastUnpacked()));
        if (lastInstalledPackageOptional.isPresent()) {
            JcrPackage lastInstalledPackage = lastInstalledPackageOptional.get();
            PackageDefinition lastInstalledPackageDefinition = new PackageDefinition(lastInstalledPackage);
            PackageDefinition thisPackageDefinition = new PackageDefinition(jcrPackage);
            if (lastInstalledPackageDefinition.getLastUnpacked() == null) {
                return false;
            }
            return lastInstalledPackageDefinition.hasSamePid(thisPackageDefinition);
        }
        return false;
    }

    private boolean isPackageOldEnough(JcrPackage jcrPackage, Integer maxAgeInDays) throws RepositoryException, IOException {
        Period maxAge = Period.ofDays(maxAgeInDays);
        LocalDate oldestAge = LocalDate.now().minus(maxAge);
        Calendar packageCreatedAtCalendar = jcrPackage.getPackage().getCreated();
        try {
            if (packageCreatedAtCalendar == null && (packageCreatedAtCalendar = jcrPackage.getDefinition().getNode().getProperty("jcr:created").getValue().getDate()) == null) {
                LOG.warn("Package [ {} ] has no created date, assuming it's NOT old enough", (Object)jcrPackage.getNode().getPath());
                return false;
            }
        }
        catch (RepositoryException e) {
            LOG.error("Unable to get created date for package [ {} ]", (Object)jcrPackage.getNode().getPath(), (Object)e);
            return false;
        }
        LocalDate packageCreatedAt = LocalDateTime.ofInstant(packageCreatedAtCalendar.toInstant(), packageCreatedAtCalendar.getTimeZone().toZoneId()).toLocalDate();
        String packageDescription = this.getPackageDescription(jcrPackage);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Checking if package is old enough: Name: {}, Created At: {}, Oldest Age: {}", new Object[]{packageDescription, packageCreatedAt.format(LOCALIZED_DATE_FORMATTER), oldestAge.format(LOCALIZED_DATE_FORMATTER)});
        }
        return !packageCreatedAt.isAfter(oldestAge);
    }

    private String getPackageDescription(JcrPackage jcrPackage) throws RepositoryException {
        JcrPackageDefinition definition = jcrPackage.getDefinition();
        Node packageNode = jcrPackage.getNode();
        if (definition != null && packageNode != null) {
            return String.format("%s:%s:v%s [%s]", definition.getId().getName(), definition.getId().getGroup(), definition.getId().getVersionString(), packageNode.getPath());
        }
        return "Unknown package";
    }

    static class PackageDefinition {
        JcrPackage jcrPackage;

        public PackageDefinition(@Nonnull JcrPackage jcrPackage) {
            this.jcrPackage = jcrPackage;
        }

        public Calendar getLastUnpacked() {
            try {
                JcrPackageDefinition definition = this.jcrPackage.getDefinition();
                if (definition != null) {
                    return definition.getLastUnpacked();
                }
                return null;
            }
            catch (RepositoryException ex) {
                return null;
            }
        }

        public boolean isSameNameAndGroup(JcrPackage otherPackage) {
            Optional<PackageId> otherPackageId = this.getPid(otherPackage);
            Optional<PackageId> thisPackageId = this.getPid(this.jcrPackage);
            if (otherPackageId.isPresent() && thisPackageId.isPresent()) {
                return otherPackageId.get().getGroup().equals(thisPackageId.get().getGroup()) && otherPackageId.get().getName().equals(thisPackageId.get().getName());
            }
            return false;
        }

        public PackageId getId() {
            try {
                JcrPackageDefinition definition = this.jcrPackage.getDefinition();
                if (definition != null) {
                    return definition.getId();
                }
                return null;
            }
            catch (RepositoryException ex) {
                return null;
            }
        }

        private Optional<PackageId> getPid(JcrPackage jcrPkg) {
            try {
                return Optional.ofNullable(jcrPkg.getDefinition()).map(JcrPackageDefinition::getId);
            }
            catch (RepositoryException ex) {
                return Optional.empty();
            }
        }

        public boolean hasSamePid(PackageDefinition jcrPkg) {
            try {
                Optional<PackageId> pkgId = Optional.ofNullable(jcrPkg.getId());
                return pkgId.map(packageId -> packageId.equals((Object)this.getId())).orElse(false);
            }
            catch (NullPointerException ex) {
                return false;
            }
        }
    }
}

